// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.io;

import java.io.*;
import java.util.*;
import java.util.logging.*;

/** 
Subclasses {@link java.util.logging.Logger} and provides additional
convenience methods. These convenience methods primarily take
thread and object references in their method signatures (which
the <tt>java.util.logging.Logger</tt> lacks).
<p>
Some Notes about <tt>java.util.logging.Logger</tt>
<ul>
	<li>The <tt>log(..)</tt> methods are *much* slower than the 
		<tt>log<b>p</b>(...)</tt> methods. The <tt>log(..)</tt>
		methods try to intuit the class and method name from the
		call stack and hence take too much time. These methods
		should be avoided almost completely for frequent logging 
		calls that will remain enabled during production.
	<li>The level shorthand methods ( <tt>severe(..)</tt>, <tt>warn(..)</tt>)
	internally delegate to the <tt>log(..)</tt> methods and hence are quite
	slow. They can still be used however if methods such as <tt>fine(..)</tt>
	are probably going to be disabled during production. Methods such as
	<tt>severe(..)</tt> are not expected to be called too often during
	production and hence can remain enabled at production.
	<li>java.util.logging.Handlers assigned to different loggers (even for 
	parent vs. child) should be different objects. If the same handler object is 
	instantiated and assigned to more than one Logger, then the LogManager's
	<tt>reset</tt> method will attempt to remove/close the same handler more 
	than once. This may happen at JVM shutdown, for example. Since this class
	writes a shutdown notice to each handler, if the same handler is used by 
	more than one FCLogger, than the shutdown notice will not be written
	properly for each FCLogger (because the shared handler may have been
	closed previously). It is therefore recommended to use different
	handlers for different loggers, even if those different handlers 
	point to the same file or destination. 
</ul>
<p>
ClassLoader issues:<br>
We need to separate our Logger instances, if these instances are
used or loaded by separate classloaders (for example, by more than
1 webapp or virtual host or the same webapp when reloaded). We
<u>cannot</u> register two instances of a logger with the same
name in the {@link java.util.logging.LogManager} class because if
we did, it would mean FCLogger created by 2 separate classloaders
would be registered (and retrieved) under the same name from a
system class (the system LogManager) which would then result in
<tt>ClassCastExceptions</tt>
<p>
We can either not use the LogManager at all (use our own substitute) or 
simply separate our Logger instances into different namespaces. Using
our own subsitute LogManager (which would be seperately loaded by
different classloaders) doesn't really work because a Logger itself
uses the Java LogManager internally. We really do need to come up
with a convention to use a seperate namespace per application. We
can do this by either prepending the classloader to the logger's name
or prepending a unique application specific name to the logger.
<p> 
Another approach is to always delegate to the system classloader
to install/register all loggers that are created. 
<p>
Thread Safety: This class <tt>is</tt> Threadsafe and all it's methods 
can be used concurrently.

@author 	hursh jain
@version 	1.4  05/24/2002
@see 		fc.app.App
**/

public class FCLogger extends java.util.logging.Logger 
{
/**
<tt>true</tt> if currently debugging, <tt>false</tt> otherwise. Not 
necessary  for simple debugging statements of the form:
<blockquote>
	mylog.log(Level.FINE, "abc");
</blockquote>
because all <tt>bug</tt> methods simply return if this variable is
false. But for complex statements, the following is suggested:
<blockquote>
if (mylog.on) {
	mylog.log("abc" + some-variable + ";" + some-other-variable)
	}
</blockquote>
This avoids needless string creation/concatenation is logging
is turned off. But note this will still entail the needless 
creation if logging is set to say Level.CONFIG, even though the
message itself (at Level.FINE) will never be printed. It's best
to use the {@link java.util.logging#isLoggable} method or it's
more convenient form implemented in this class {@link #go}
**/
public volatile boolean	on; 

/**
Constructs a Logger. See {@link java.util.logging.Logger} for more
information.
<p>
**/	
public FCLogger(String name, String resourceBundleName)
	{
	super(name, resourceBundleName);
	on = true;  //java.util.logging.Logger is initially *not* OFF.
	}

/** 
Find or create a logger for a named subsystem.  If a logger has
already been created with the given name it is returned.  Otherwise
a new FClogger is created/<p>
<b>Note</b>: This method always returns an object whose run time type
is <tt>FCLogger</tt>. If a logger with the specified name already
exists (say it was created previously) and that existing logger is
<b>not</b> of type <tt>FCLogger</tt>, then a runtime
<tt>ClassCastException</tt> will be thrown. This can happen if the
existing logger was created with a different method (say directly via
{@link java.util.logging.Logger#getLogger}). This class depends upon and
uses the {@link java.util.logging.LogManager}.  It is suggested
that all loggers in the application be created via this or the
{@link #getFCLogger} methods only.
@param	name		A name for the logger. Should be unique across
					for loggers used by different classloaders. 

@return a suitable FCLogger. 
*/
public static synchronized Logger getLogger(String name) 
	{
	LogManager manager = LogManager.getLogManager();
	Logger result = manager.getLogger(name);
	if (result == null) {
	    result = new FCLogger(name, null);
	    manager.addLogger(result);
	    result = manager.getLogger(name);
		}
	return (FCLogger) result;
	}
	
/**
Similar to {@link #getLogger}, except it returns a <tt>FCLogger</tt>
and therefore, the calling client does not have to perform any
additional casts.

@param	name		A name for the logger. Should be unique across
					for loggers used by different classloaders. 
**/
public static synchronized FCLogger getFCLogger(String name) 
	{
	LogManager manager = LogManager.getLogManager();
	/*
	System.out.println(
		"FCLogger.class@=" + System.identityHashCode(FCLogger.class) + 
		"; FCLogger.classloader[" + FCLogger.class.getClassLoader() + "]@=" + System.identityHashCode(FCLogger.class.getClassLoader()) + 
		"; LogManager@=" + System.identityHashCode(LogManager.class) + 
		"; LogManager.classloader["  + LogManager.class.getClassLoader() + "]@" + 
			System.identityHashCode(LogManager.class.getClassLoader()));
	*/
	FCLogger result = (FCLogger) manager.getLogger(name);
	if (result == null) {
	    result = new FCLogger(name, null);
	    manager.addLogger(result);
		}
	return result;
	}

/** See {@link java.util.logging.Logger#addHandler} */
public void addHandler(Handler handler) throws SecurityException
	{
	super.addHandler(handler);
	LogRecord rec = new LogRecord(Level.INFO,
						"Log [" + getName() + "] opened on: " + new Date());
	handler.publish(rec);
	}	

/** A more convenient name than {@link java.util.logging#isLoggable} **/	
public boolean go(Level level) {
	return isLoggable(level);
	}	
	
	
/** See {@link java.util.logging.Logger#removeHandler} */
public void removeHandler(Handler handler) throws SecurityException
	{
	//System.out.println("removeHandler: " + handler);
	//new Exception().printStackTrace();
	Handler[] handlers = getHandlers();
	if ( Arrays.asList(handlers).contains(handler) ) {
		LogRecord rec = new LogRecord(Level.ALL,
				"Log [" + getName() + "] closed on: " + new Date());
		handler.publish(rec);
		super.removeHandler(handler);
		}
	}	
	

/** Removes all handlers from this logger. Does <b>not</b> publish a
log closed statement to each handler while doing so
*/
public void removeAllHandlers()
	{
	Handler[] handlers = getHandlers();
	for (int n = 0; n < handlers.length; n++) {
		super.removeHandler(handlers[n]);
		}
	}	
	
public void setLevel(Level newLevel) throws SecurityException {
	super.setLevel(newLevel);
	if (Level.OFF.equals(newLevel)) 
		on = false;
	else 
		on = true;
	}

/**
Calls <tt>flush()</tt> on all handlers attached to this logger.
**/
void flush() 
	{
	try { 
		Handler[] hands =  getHandlers();
		for (int n = 0; n < hands.length; n++)
			hands[n].flush();			
		}
	catch (Exception e) { 
		/*ignore*/ 
		}
	}

public String toString()
	{
	String result =  getClass().getName() + 
		"; Level=" + getLevel();
	return result;
	}

//============= convenience methods ======================

/** 
Additional convenience method 
@param 	level	one of the message level identifiers
@param  thread	calling Thread 
@param	caller	calling Object
@param 	msg		a message String 
@param	param	an array of Object[]
**/
public void log(Level level, Thread thread, Object caller, String msg, Object[] params)
	{
	logp(level, (thread!=null)?thread.toString():null, 
					(caller!=null)?caller.toString():null, msg, params);
	}

/** 
Additional convenience method 
@param 	level		one of the message level identifiers
@param  thread		calling Thread 
@param	caller		calling Object
@param 	msg			a message String 
@param	thrown		Throwable associated with log message.
**/
public void log(Level level, Thread thread, Object caller, String msg, Throwable thrown)
	{
	logp(level, (thread!=null)?thread.toString():null, 
					(caller!=null)?caller.toString():null, msg, thrown);
	}

/** 
Additional convenience method 
@param 	level		one of the message level identifiers
@param  thread		calling Thread 
@param	caller		calling Object
@param 	msg			a message String 
**/
public void log(Level level, Thread thread, Object caller, String msg) 
	{
	logp(level, (thread!=null)?thread.toString():null, 
					(caller!=null)?caller.toString():null, msg);
	}

/** 
@param 	level	one of the message level identifiers
@param 	msg		a message String 
**/	
public void log(Level level, String msg)
	{
	logp(level, null, null, msg);	
	}

/** 
Logs an exception stack trace at <tt>Level.WARNING</tt>
**/	
public void log(Throwable e)
	{
	logp(Level.WARNING, null, null, "", e);	
	}


}   //~class Logger


