// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.jdbc;

import java.io.*;
import java.util.*;
import java.sql.*;

import fc.io.*;
import fc.util.*;

/** 
This class implements the gateway to a SQL Database. It should be
used to request connections.

@author hursh jain
*/
public abstract class ConnectionMgr
{
final protected 	Log 		log;  //for all logging messages
					String	 	myname = getClass().getName();	

private	  volatile 	boolean	 	mgrClosed = false;
//== jdbc ====================================
protected volatile 	String		catalogName;
protected 			String 		url;
protected 			String		driver;
protected 			String	 	user;
protected 			String		password;
protected 			Driver	 	mydriver;
protected			DBName		dbname;

private static class PropertyMgrWrapper extends PropertyMgr
	{
	Map map = new HashMap();
	PropertyMgrWrapper(
		String url, String driver, 
		String user, String password, String catalog)
		{
		map.put("jdbc.url", url);
		map.put("jdbc.driver", driver);
		map.put("jdbc.user", user);
		map.put("jdbc.password", password);
		map.put("jdbc.catalog", catalog);
		}
	
	public String get(String name) {
		return (String) map.get(name);
		}		
	public String set(String name, String backup) { Log.getDefault().error("not implemented"); return null; }
	public void save() { Log.getDefault().error("not implemented");}
	}

/**
Creates a new ConnectionMgr with logging to
logging to {@link fc.io.Log#getDefault()}. 

@param  jdbc.url
@param  jdbc.driver
@param  jdbc.user
@param  jdbc.password
@param  jdbc.catalog    optional, sets the default and can be
<tt>null</tt>
*/
public ConnectionMgr(
		String jdbc_url, String jdbc_driver, 
		String jdbc_user, String jdbc_password, 
		String jdbc_catalog) 	
	throws Exception
	{	
     //we need to delegate because other constructrs do
	 //things like driver initialization etc.
	this(new PropertyMgrWrapper(
		jdbc_url, jdbc_driver, jdbc_user, jdbc_password, 
		jdbc_catalog));
	}

/**
Delegates to {@link ConnectionMgr(Log, PropertyMgr)} with logging to {@link fc.io.Log#getDefault()}.
*/
public ConnectionMgr(PropertyMgr props) 
throws Exception
	{
	this(Log.getDefault(), props, null);	
	}

/**
Delegates to {@link ConnectionMgr(Log, PropertyMgr)} with logging to {@link fc.io.Log#getDefault()}
and using the specified <tt>prefix</tt> for property names.
*/
public ConnectionMgr(PropertyMgr props, String prefix) 
throws Exception
	{
	this(Log.getDefault(), props, prefix);	
	}

/**
Constructs a new ConnectionMgr. The default implementation does not use a
connection pool but creates new Connections, whenever needed. Expects the
following properties, which should accessible via the specified {@link
fc.app.PropertyMgr}.
<ol>
	<li><tt>jdbc.url</tt></li>
	<li><tt>jdbc.driver</tt></li>
	<li><tt>jdbc.user</tt></li>
	<li><tt>jdbc.password</tt></li>
	<li>An optional <tt>jdbc.catalog</tt> property sets the
	default jdbc catalog. Can be <tt>null</tt>.</li>
</ol>
Throws an Exception if this object cannot be constructed for some reason.
**/
public ConnectionMgr(Log log, PropertyMgr props) 
throws Exception
	{
	this(log, props, null);
	}


/**
Constructs a new ConnectionMgr. The default implementation does not use a
connection pool but creates new Connections, whenever needed. Expects the
following properties, which should accessible via the specified {@link
fc.app.PropertyMgr}.
<ol>
	<li><tt>jdbc.url</tt></li>
	<li><tt>jdbc.driver</tt></li>
	<li><tt>jdbc.user</tt></li>
	<li><tt>jdbc.password</tt></li>
	<li>An optional <tt>jdbc.catalog</tt> property sets the
	default jdbc catalog. Can be <tt>null</tt>.</li>
</ol>
The optional prefix parameter (if not null) is prepended to each property name
before it is accessed from the property manager.
For example:
	<blockquote><tt>jdbc.url</tt></blockquote>
becomes:
	<blockquote><i>prefix</i><tt>jdbc.url</tt></blockquote>

Throws an Exception if this object cannot be constructed for some reason.
**/
public ConnectionMgr(Log log, PropertyMgr props, String prefix) 
throws Exception
	{
	//1. don't call toString() from this constructor since it's
	//overriden in subclasses
	
	Argcheck.notnull(log); 
	Argcheck.notnull(props);
	
	this.log = log;
	
	url 		= 	props.getRequired(prefix!=null?prefix+"jdbc.url":"jdbc.url");
	driver 		= 	props.getRequired(prefix!=null?prefix+"jdbc.driver":"jdbc.driver");
	user 		= 	props.getRequired(prefix!=null?prefix+"jdbc.user":"jdbc.user");
	password 	= 	props.getRequired(prefix!=null?prefix+"jdbc.password":"jdbc.password");
	catalogName	= 	props.get(prefix!=null?prefix+"jdbc.catalog":"jdbc.catalog"); //can be null;
	
	// Register driver, Use normal instantiation and manually register the
   	//driver - does not rely on static initializer being run.
   	final Class mydriver_class = Class.forName(driver);
	mydriver = (Driver) mydriver_class.newInstance();
	DriverManager.registerDriver(mydriver);
	
	dbname = DBName.fromDriver(mydriver_class.getName());
	if (dbname == null) {
		log.warn("Cannot guess dbname from the driver. Connections will still work perfectly but getDBName() in this class will return null");
		}
		
	String sep = System.getProperty("line.separator");
	log.bug("ConnectionMgr constructor; driver="+driver+sep+"url="+url+sep+"user="+user+sep+"password="+password+sep+"catalog="+catalogName);	
	}


/** 
Returns the {@link java.sql.Driver} that this connection manager
is using to connect to the database.
**/
public Driver getDriver() {
	return mydriver;
	}

/**
Returns the dbname corresponding that the database connected to by this
connection manager. Useful for writing database specific code as/when
applicable.
*/
public DBName getDBName() throws SQLException
	{
	return dbname;
	}

/**
If set, the ConnectionMgr will always return connections
set to the specified catalog. Set this to <tt>null</tt> to
return connections not set to any catalog.
**/	
public void setCatalog(String name) {
	this.catalogName = name;
	}

/**
Returns a connection if successful, otherwise throws a
<tt>SQLException</tt>. The returned has the default JDBC
properties, including being in auto-commit mode (which means
that it automatically commits changes after executing each
statement).
**/
public Connection getConnection() throws SQLException
	{	
	if (mgrClosed) {
		log.warn(this, "tried to get a connection from closed ConnectionMgr");
	    throw new SQLException("tried to get a connection from closed ConnectionMgr");
		}
	Connection con = getConImpl();
	if (catalogName != null) {
		log.bug(this, "setting catalog name to: ", catalogName);
		con.setCatalog(catalogName);
		}
	log.bug(this, ", returning connection=", con);
	return con;
	}

/**
Closes the ConnectionMgr. After this call, no more connections can be checked
out and any existing checked out connections are closed. Does <b>not</b> throw
an Exception but returns false (instead) if the ConnectionMgr could not 
be closed. Calls the {@link #handleMgrShutdown()} method inside a 
synchronized block.

@return 	true if connection manager closed successfully, false otherwise
**/
synchronized public boolean close() 
	{
	synchronized (this) {
		mgrClosed = true;
		log.bug(this, ": closing ConnectionMgr");
		return handleMgrShutdown();
		}
	}
		
/** Returns the jdbc url that this connection manager is using, **/
public String getURL() {
	return url;
	}		
		
public String toString() {
	return myname;
	}	

//== Abstract methods ======================================

/** 
Is called by the {@link #close()} method. By default does nothing but
subclasses can override this method to handle connection manager 
shutdown as deemed necessary.
@return	true if handler succeeded, false otherwise. 
**/
protected abstract boolean handleMgrShutdown();
	
/**
Returns a connection to the DB everytime it's called.
**/	
protected abstract Connection getConImpl() throws SQLException;

}  //~class ConnectionMgr


