// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.jdbc;

import java.sql.*;
import java.util.*;

import fc.io.*;
import fc.util.*;

/** 
A poolable {@link java.sql.Connection} that works with the {@link
PooledConnectionMgr}. Closing this connection marks it as closed and
returns it to the pool. 
<p>
The connection can be retrieved and closed as many times as needed.
It is very important that a connection be closed after use (this
is true for all connections, pooled or regular).
<p>
Typically one does so by saying:
<blockquote>
<pre>
Connection con;
try {
	con = [..get either a pooled or regular connection..]
	..use..
	<b>con.commit();</b>  //if transactional
	}
catch (SQLException e) {
	<b>con.rollback();</b> //if transactional
	}
finally {
	<b>con.close();</b>  //always (transactional or not)
	}
</pre>
</blockquote>
<b>Super Important</b>: transactions are not rolled back or committed when <code>close</code> is
invoked. You must either commit or rollback the transaction if in transaction mode (ie you
have called <code>setAutoCommit(false)</code> on this connection). Returning the connection (via close)
back to the pool without committing/rolling-back will keep the prior statements around and those will
be committed/rolledback at a later date if commit/rollback is invoked upon retrieving/using the connection
again.

@author hursh jain
**/
public class PooledConnection implements Connection
{
static int idCounter = 1;

//each pooled connection has an ID which is useful to track
//check in/check outs from the pool. id's are assigned from
//the idCount variable
int 					id;		
PooledConnectionMgr 	pool;  	//the parent pool
Connection 				connection;
volatile boolean 		closed;
//statistical data
int						commitCount;
int						rollbackCount;
int						warningsCount;
long					transaction_start_time = 0;

/** 
Creates a initially open pooled connection that wraps the
specified jdbc connection.
**/
public PooledConnection(PooledConnectionMgr pool, Connection con) 
	{
	Argcheck.notnull(pool, "specified 'pool' arg was null");
	Argcheck.notnull(con, "specifed 'con' arg was null");
	this.pool = pool;
	this.connection = con;
	this.closed = false;
	id = makeID(); 
	}

private static final synchronized int makeID() {
	return idCounter++;
	}

protected int getID() {
	return id;
	}

/** 
<tt>true</tt> to mark this connection as closed, <tt>false</tt>
to mark it as open.
**/
void setClosed(boolean val) 
	{
	closed = val;
	}
	
void checkOpen() throws SQLException 
	{
	if (closed) {
		throw new SQLException("This connection has been closed");
		}
    }
	
//== pooled  =========================================

/**
transactions are not rolled back or committed when <code>close</code> is invoked. You must either commit or 
rollback the transaction if in transaction mode (ie you have called <code>setAutoCommit(false)</code> on this 
connection). 
<p>
eturning the connection (via close) back to the pool without committing/rolling-back will keep 
the prior statements around and those will be committed/rolledback at a later date if commit/rollback is 
invoked upon retrieving/using the connection again.
*/
public void close() throws SQLException 
	{
	if (! isClosed() ) {
		//pool.log.bug("closing connection", IOUtil.throwableToString(new Exception("Debug Stack Trace")));
		setClosed(true);
		pool.put(this);
		}
	else {
		pool.log.bug("Tried to close already closed connection (#", id, ") =>", IOUtil.throwableToString(new Exception("Debug Stack Trace")));
		}
	}

void closeReally() throws SQLException 
	{
	connection.close();
	}


public boolean isClosed() {
    return closed;
	}

//we must use interned strings with id based hashmaps;
private final Map pstmtMap = new IdentityHashMap(256);
public PreparedStatement getCachedPreparedStatement(String sql) 
throws SQLException
	{
	sql = sql.intern();
	synchronized (this)
		{
		//intern can be tricky. we must use the string
		//*returned* by the intern() method
		if (pstmtMap.containsKey(sql)) {
			final PreparedStatement ps = (PreparedStatement) pstmtMap.get(sql);
			ps.clearParameters();
			return ps;
			}
		else{
			final PreparedStatement pstmt = this.prepareStatement(	
						sql,
						ResultSet.TYPE_SCROLL_INSENSITIVE,
						ResultSet.CONCUR_READ_ONLY);
			pstmtMap.put(sql, pstmt);
			return pstmt;
			}
		}
	}

public String toString() {
	StringBuilder buf = new StringBuilder();
	buf.append(super.toString());
	buf.append("/PoolID:");
	buf.append(id);
	try {
		buf.append("/currentIsolation:");
		int i = getTransactionIsolation();
		switch (i)
			{
			case Connection.TRANSACTION_NONE: 
				buf.append("NONE"); break;
			case Connection.TRANSACTION_READ_UNCOMMITTED : 
				buf.append("READ_UNCOMMITTED"); break;
			case Connection.TRANSACTION_READ_COMMITTED: 
				buf.append("READ_COMMITTED"); break;
			case Connection.TRANSACTION_REPEATABLE_READ : 
				buf.append("REPEATABLE_READ"); break;
			case Connection.TRANSACTION_SERIALIZABLE : 
				buf.append("SERIALIZABLE"); break;
			default:				
				buf.append("Unknown: ");
				buf.append(i);
			}
		buf.append("/autoCommit:");
		buf.append(String.valueOf(getAutoCommit()).toUpperCase());
		if (transaction_start_time == 0) {
			buf.append("/[no open transaction]");
			}
		else{
			buf.append("/[in open transaction,");
			buf.append(System.currentTimeMillis() - transaction_start_time);
			buf.append(" ms]");
			}
		}
	catch (SQLException e) {
		}
	return buf.toString();
	}




//== wrapped methods ====================================

public void clearWarnings() throws SQLException {
    connection.clearWarnings();
	}

public void commit() throws SQLException 
	{
    commitCount++;
	transaction_start_time = 0;
    connection.commit();
	}

public Statement createStatement() throws SQLException {
    return connection.createStatement();
	}

public Statement createStatement(int resultSetType, 
                                 int resultSetConcurrency) throws SQLException {
    
    return connection.createStatement(resultSetType, resultSetConcurrency);
	}

public boolean getAutoCommit() throws SQLException {
    return connection.getAutoCommit();
	}

public String getCatalog() throws SQLException {
    return connection.getCatalog();
	}

public DatabaseMetaData getMetaData() throws SQLException {
    return connection.getMetaData();
	}

public int getTransactionIsolation() throws SQLException {
    return connection.getTransactionIsolation();
	}

public Map getTypeMap() throws SQLException {
    return connection.getTypeMap();
	}

public SQLWarning getWarnings() throws SQLException {
    warningsCount++;
    return connection.getWarnings();
	}


public boolean isReadOnly() throws SQLException {
    return connection.isReadOnly();
	}

public String nativeSQL(String sql) throws SQLException {
    return connection.nativeSQL(sql);
	}    

public CallableStatement prepareCall(String sql) throws SQLException {
    return connection.prepareCall(sql);
	}

public CallableStatement prepareCall(String sql, int resultSetType, 
                                     int resultSetConcurrency) throws SQLException {
    return connection.prepareCall(sql, resultSetType, resultSetConcurrency);
	}

public PreparedStatement prepareStatement(String sql) throws SQLException {
    return connection.prepareStatement(sql);
	}

public PreparedStatement prepareStatement(String sql, int resultSetType, 
                                          int resultSetConcurrency) throws SQLException {
    return connection.prepareStatement(sql, resultSetType, resultSetConcurrency);
	}

public void rollback() throws SQLException {
	connection.rollback();
	}

public void setAutoCommit(boolean b) throws SQLException  
	{
	if (b)
   		transaction_start_time = 0;
	else
		transaction_start_time = System.currentTimeMillis();

   	connection.setAutoCommit(b);
	}
		
public void setCatalog(String catalog) throws SQLException {
    connection.setCatalog(catalog);
	}

public void setReadOnly(boolean readOnly) throws SQLException {
    connection.setReadOnly(readOnly);
	}

public void setTransactionIsolation(int level) throws SQLException {
    connection.setTransactionIsolation(level);
	}

public void setTypeMap(Map map) throws SQLException {
    connection.setTypeMap(map);
	}

public int getHoldability() throws SQLException {
    return connection.getHoldability();
	}

public void setHoldability(int holdability) throws SQLException {
    connection.setHoldability(holdability);
	}

public java.sql.Savepoint setSavepoint() throws SQLException {
    return connection.setSavepoint();
	}

public java.sql.Savepoint setSavepoint(String name) throws SQLException {
    return connection.setSavepoint(name);
	}

public void rollback(java.sql.Savepoint savepoint) throws SQLException {
    rollbackCount++;
    connection.rollback(savepoint);
	}

public void releaseSavepoint(java.sql.Savepoint savepoint) throws SQLException {
    connection.releaseSavepoint(savepoint);
	}

public Statement createStatement(int resultSetType,
                                 int resultSetConcurrency,
                                 int resultSetHoldability) throws SQLException  {
    return connection.createStatement(resultSetType, resultSetConcurrency, resultSetHoldability);
	}

public Struct createStruct(String str, Object[] arr) throws SQLException  {
    return connection.createStruct(str, arr);
	}

public java.util.Properties getClientInfo() throws SQLException {
	return connection.getClientInfo();
	}

public String getClientInfo(String name) throws SQLException {
	return connection.getClientInfo(name);
	}

public void setClientInfo(String key, String val) throws SQLClientInfoException {
	connection.setClientInfo(key, val);
	}

public void setClientInfo(java.util.Properties props) throws SQLClientInfoException {
	connection.setClientInfo(props);
	}
                         
public java.sql.Array createArrayOf(String str, Object[] arr) throws SQLException  {
    return connection.createArrayOf(str, arr);
	}

public Blob createBlob() throws SQLException {
	return connection.createBlob();
	}

public Clob createClob() throws SQLException {
	return connection.createClob();
	}

public NClob createNClob() throws SQLException {
	return connection.createNClob();
	}

public SQLXML createSQLXML() throws SQLException  {
	return connection.createSQLXML();
	}
                
public boolean isValid(int timeout) throws SQLException {
	return connection.isValid(timeout);
	}

public boolean isWrapperFor(Class iface) throws SQLException {
	return connection.isWrapperFor(iface);
	}

public Object unwrap(Class iface) throws SQLException {
	return connection.unwrap(iface);
	}

public PreparedStatement prepareStatement(String sql, int resultSetType,
                                          int resultSetConcurrency,
                                          int resultSetHoldability) throws SQLException  {
    return connection.prepareStatement(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
	}

public CallableStatement prepareCall(String sql, int resultSetType,
                                     int resultSetConcurrency,
                                     int resultSetHoldability) throws SQLException {
    return connection.prepareCall(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
	}

public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
		return connection.prepareStatement(sql, autoGeneratedKeys);
	}

	public PreparedStatement prepareStatement(String sql, int columnIndexes[]) throws SQLException {
		return connection.prepareStatement(sql, columnIndexes);
	}

public PreparedStatement prepareStatement(String sql, String columnNames[]) throws SQLException {
	return connection.prepareStatement(sql, columnNames);
	}

//====================== 1.7 + ===============================
/* enable this when building for jdk7+ */
/*
public int getNetworkTimeout() 
throws SQLException {
	return connection.getNetworkTimeout();
	}
	
public void setNetworkTimeout(java.util.concurrent.Executor executor, int milliseconds) 
throws SQLException {
	connection.setNetworkTimeout(executor, milliseconds);
	}	

public void abort(java.util.concurrent.Executor executor)
throws SQLException {
	connection.abort(executor);
	}

public String getSchema()
throws SQLException {
	return connection.getSchema();
	}
          
public void setSchema(String schema)
throws SQLException {
	connection.setSchema(schema);
	}
*/
} //~PooledConnection
