// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.jdbc.dbo;

import java.sql.*;
import java.util.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/**
Fixes class and method names based on various 'wrangle' properties from
the configuration file.

@author hursh jain
*/
class NameWrangle
{
Log 			log;
//settings from config file
NameWrangle.Type	wrangle_classname;
NameWrangle.Type	wrangle_methodname;
String				wrangle_sentence_delimiter;

public NameWrangle(PropertyMgr props, Log log)
	{
	this.log = log;
	
	String keyname = "generate.wrangle.classname";
	
	String temp = props.get(keyname, "none");
	//default: none
	log.bug(keyname, "=", temp);
	try {
		wrangle_classname = Type.parseString(temp);
		}
	catch (RuntimeException e) {
		log.error("make sure '", keyname, "' is specified properly");
		throw e;
		}
		
	if (wrangle_classname == Type.NONE_UNDER)
		throw new IllegalArgumentException("none_under is not a valid option for 'generate.wrangle.classname'");
		
	keyname = "generate.wrangle.methodname";
	//default: none_under
	temp = props.get(keyname, "none_under");
	log.bug(keyname, "=", temp);
	try {
		wrangle_methodname = Type.parseString(temp);	
		}
	catch (RuntimeException e) {
		log.error("make sure '", keyname, "' is specified properly");
		throw e;
		}
		
	//default: "_"
	wrangle_sentence_delimiter = 
		props.get("generate.wrangle.sentence.delimiter", "_");
	}

String getClassName(Table table)	
	{	
	return getClassName(table.getName());
	}
	
String getGetName(ColumnData col)  
	{
	return getWrangledMethodName(col.getName(), "get");	
	}

String getSetName(ColumnData col) 
	{
	return getWrangledMethodName(col.getName(), "set");
	}

String getSetNullName(ColumnData col) 
	{
	return getWrangledMethodName(col.getName(), "setNull");
	}
	
String getIsModifiedName(ColumnData col)
	{
	return getWrangledMethodName(col.getName(), "isModified");
	}

String getIsModifiedNullName(ColumnData col) 
	{
	return getWrangledMethodName(col.getName(), "isModifiedSetNull");
	}
	

String getIsNullInDBName(ColumnData col)
	{
	return getWrangledMethodName(col.getName(), "isNullInDB");
	}

String getIncName(ColumnData col)  
	{
	return getWrangledMethodName(col.getName(), "inc");	
	}

String getDecName(ColumnData col) 
	{
	return getWrangledMethodName(col.getName(), "dec");
	}


String getClassName(String tablename) 
	{
	String name = null;
	if (wrangle_classname == Type.NONE) {
		name = tablename;			
		}
	else if (wrangle_classname == Type.UPPER) {
		name = tablename.toUpperCase();
		}
	else if (wrangle_classname == Type.LOWER) {
		name = tablename.toLowerCase();
		}
	else if (wrangle_classname == Type.SENTENCE) {
		name = StringUtil.sentenceCase(tablename, wrangle_sentence_delimiter);
		}
	else if (wrangle_classname == Type.FIRSTUP) {
		name =  
			tablename.substring(0, 1)
					 .toUpperCase()
			 +
			tablename.substring(1, tablename.length())
					 .toLowerCase();
		}
	else {	
		log.error("this code should not be reached");
		throw new IllegalArgumentException("An internal error occured");
		}
	return name;
	}
	
//Implementation methods	
	
String getWrangledMethodName(String colname, String prefix)
	{
	String name = null;
	if (wrangle_methodname == Type.NONE) {
		name = prefix + colname;			
		}
	else if (wrangle_methodname == Type.NONE_UNDER) {
		name = prefix + "_" + colname;
		}		
	else if (wrangle_methodname == Type.UPPER) {
		name = prefix + colname.toUpperCase();
		}
	else if (wrangle_methodname == Type.LOWER) {
		name = prefix + colname.toLowerCase();
		}
	else if (wrangle_methodname == Type.SENTENCE){
		name = prefix + 
				StringUtil.sentenceCase
					(colname, wrangle_sentence_delimiter);
		}
	else if (wrangle_methodname == Type.FIRSTUP) {
		name = 	prefix +
				colname.substring(0, 1)
						 .toUpperCase() 
				 +
				colname.substring(1, colname.length())
						 .toLowerCase();
			}
	else {
		log.error("this code should not be reached");
		throw new IllegalArgumentException("An internal error occured");
		}
	return name;
	}

static ToString.Style style = new ToString.Style();
static {
	style.reflectVisibleLevel = ToString.Style.VisibleLevel.PRIVATE;
	}
public String toString()  {
	return new ToString(this, style).reflect().render();
	}


static class Type
	{
	final String  	typedesc;
	final int		typenum;
	
	private Type(String typedesc, int typenum) {
		this.typedesc = typedesc;
		this.typenum = typenum;
		}
	
	public static final Type NONE 	  = new Type("none", 0);
	public static final Type NONE_UNDER = new Type("none_under", 1);
	public static final Type UPPER 	  = new Type("upper", 2); 
	public static final Type LOWER	  = new Type("lower", 3);	
	public static final Type SENTENCE = new Type("sentence", 4);
	public static final Type FIRSTUP  = new Type("firstup", 5);

	public static Type parseString(String typedesc)
		{
		Argcheck.notnull(typedesc, "typedesc paramater was null");
		Type type = null;
		typedesc = typedesc.toLowerCase().intern();
					
		if (typedesc == "none")
			type = NONE;
		else if (typedesc == "none_under")
			type = NONE_UNDER;		
		else if (typedesc == "upper")
			type = UPPER;
		else if (typedesc == "lower")
			type = LOWER;
		else if (typedesc == "sentence")
			type = SENTENCE;
		else if (typedesc == "firstup")
			type = FIRSTUP;
		else 
			throw new IllegalArgumentException("wrangle type: [" + typedesc + "] could not be parsed");
		
		//System.out.println("typedesc=" + typedesc + ";parsed="+type);
		return type; 
		} //~parsefromstring
		
	public String toString() {
		return "NameWrangle.Type='" + typedesc + "'";		
		}
	} //~class type

public static void main(String args[]) {
	final Args myargs = new Args(args);
	
	myargs.setUsage("java fc.jdbc.dbo.NameWrangle\n" 
			+ "	-teststr string_to_be_wrangled\n"
			+ "The following are optional\n" 
			+ "	-generate.wrangle.classname  valid_value_allowed_by_generate_config_file\n"
			+ "	-generate.wrangle.methodname valid_value_allowed_by_generate_config_file\n"
			+ "	-generate.wrangle.sentence.delimiter valid_value_allowed_by_generate_config_file\n"
			);
	
	String teststr = myargs.getRequired("teststr");
	
	final SystemLog log = Log.getDefault();
	log.setLevel(SystemLog.DEBUG);
	
	PropertyMgr propmgr = new PropertyMgr() 
		{
		public String get(String name) {
			return myargs.get(name);
			}
		public void save() { log.error("this should not be called"); }
		public String set(String a, String b) { log.error("this should not be called"); return null; }		
		};

	NameWrangle wrangle = new NameWrangle(propmgr, log);
	
	//log.bug("constructed new wrangle", wrangle);
	System.out.println();	
	System.out.println("getClassName=" + wrangle.getClassName(teststr));
	System.out.println("getWrangledMethodName=" + wrangle.getWrangledMethodName(teststr, "get"));
	System.out.println("getWrangledMethodName=" + wrangle.getWrangledMethodName(teststr, "set"));
	System.out.println("getWrangledMethodName=" + wrangle.getWrangledMethodName(teststr, "isModified"));
	}

}

