// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.net;

import java.io.*;
import java.util.*;
import java.util.regex.*;
import java.nio.*;
import java.nio.channels.*;
import java.net.*;

import fc.io.*;
import fc.util.*;

/**
Network Interface Card related utility methods. Many of these
methods are specific to ethernet type nic's.

@author hursh jain
**/
public class NIC 
{
//for internal debugging
private static boolean dbg = false;
static String linesep = IOUtil.LINE_SEP;

/**
Returns a Map containing an IP address as the key and
a {@link NIC.NICInfo} object as the value. Returns <tt>null</tt> if 
the platform type is not supported.
<p>
<b>Note 1:</b> Requires a NT based system (NT, 2000, XP or later) if
running on windows. (this call will not work for lesser versions of
windows).<br>
<b>Note 2:</b> An english locale is needed for this method to work
properly (because we parse for certain english words to extract the 
results).
**/
public static Map getMAC() 
	{
	Platform p = Platform.getPlatform();
	try {
		if (p == p.WINDOWS) 	return getWindowsMA();	
		else if	(p == p.LINUX) 	return getLinuxMA(); 
		else if (p == p.OSX)	 return getOSXMA(); 
	
	//we presume that linux stype ifconfig works identically
	//on other plaforms. This of course is dicey at best.
	//But I don't have no steeenkin bsd etc to test on.
	
		else if (p == p.FREEBSD) return getLinuxMA();
		else if (p == p.OPENBSD) return getLinuxMA();
		}
	catch (IOException e) {
		e.printStackTrace();
		}
	return null;
	}	

private static Map getWindowsMA() throws IOException
	{
	Map macs = new HashMap();
	Process p = Runtime.getRuntime().exec("ipconfig /all");
   	InputStream in = p.getInputStream();
   	try {
		String result = IOUtil.inputStreamToString(in, true);			
		if (dbg) System.out.println("getWindowsMA(): " + linesep + "--------------- result ------------ " + linesep + result + linesep + "--------------------------------");
		
		//for some reason ^$ does not work properly (for blank lines)
		//even when multiline mode is chosen
		
		Pattern pat = Pattern.compile(
			"(?i)(?m)(?s)Ethernet\\s+adapter\\s+(.*?):.*?" +
			"Physical\\s+Address.*?:\\s*([a-zA-Z0-9\\-]*).*?" +
			"NetBIOS.*?:\\s*\\w*"
			);

		Pattern ip_pat = Pattern.compile(
							"(?i)(?m)IP\\s+Address.*?:\\s+(.+?)$"
						 );
		Matcher matcher = pat.matcher(result);
		while (matcher.find()) {
			if (dbg) System.out.println("getWindowsMA(): found: " + matcher.group(0));
			String name = matcher.group(1);
			String mac = matcher.group(2);
			Matcher ipmatcher = ip_pat.matcher(matcher.group(0));
			while (ipmatcher.find()) {
				String ip = ipmatcher.group(1);
				if (dbg) System.out.println("ip=" + ip + "; mac=" + mac + "; name=" + name);
				NICInfo nicinfo = new NICInfo(name, mac);
				macs.put(ip, nicinfo);	
				}
			}
		} 
	finally {
		if ( in != null ) in.close();
		}
	return macs;
    }

private static Map getLinuxMA() throws IOException
	{
	Map macs = new HashMap();
	Process p = Runtime.getRuntime().exec("ifconfig");
   	InputStream in = p.getInputStream();
   	try {
		/*
		//this should give us the entire input size but does NOT
		//- instead returns 0 !
		int resultlen = in.available();  
		if (dbg) System.out.println("resultlen=" + resultlen);	
		ReadableByteChannel bytechan = Channels.newChannel(in);
		ByteBuffer bytebuf = ByteBuffer.allocate(resultlen);
		bytechan.read(bytebuf); 
		bytebuf.rewind();
		if (dbg) System.out.println("ByteBuffer=" + bytebuf);			      		
		CharBuffer cbuf = bytebuf.asCharBuffer();	
		if (dbg) System.out.println("CharBuffer=" + cbuf);			      		
		*/
		
		String result = IOUtil.inputStreamToString(in, true);

		String pat = 
		  "(?s)(?m)(?i)(eth.*?)\\s+.+?HWaddr\\s([a-zA-Z0-9:\\-]*).*?addr:(.+?)\\s"; 			

		Pattern linux_ifconfig_1_42 = Pattern.compile(pat);
		Matcher matcher = linux_ifconfig_1_42.matcher(result);
		while (matcher.find()) {
			String name = matcher.group(1);
			String mac = matcher.group(2);
			String ip = matcher.group(3);
			if (dbg) System.out.println("getLinuxMA(): found: " + matcher.group(0));
			if (dbg) System.out.println("ip=" + ip + "; mac=" + mac + "; name=" + name);
			NICInfo nicinfo = new NICInfo(name, mac);
			macs.put(ip, nicinfo);	
			}
		} 
	finally {
		if ( in != null ) in.close();
		}
	return macs;
    }


private static Map getOSXMA() throws IOException
	{
	Map macs = new HashMap();
	Process p = Runtime.getRuntime().exec("ifconfig");
   	InputStream in = p.getInputStream();
   	try {
		String result = IOUtil.inputStreamToString(in, true);
		String pat = "(?s)(?m)(?i)(en\\d+):.+?.*?status.*?$"; 			

		Pattern osx = Pattern.compile(pat);
		Matcher matcher = osx.matcher(result);

		Pattern ip_pat = Pattern.compile( "(?i)inet\\s+([0-9.]+)\\s*" );
		Pattern eth_pat = Pattern.compile( "(?i)ether\\s+([a-zA-Z0-9:\\-]*)\\s*" );

		while (matcher.find()) 
			{
			String name = matcher.group(1);

			if (dbg)  {
				System.out.println("\n\n=========found: group[0]===========");
				System.out.println(matcher.group(0));
				System.out.println("\n==================");
				}
				
			String mac = null;
			String ip = null;

			Matcher ip_matcher = ip_pat.matcher(matcher.group(0));
			Matcher eth_matcher = eth_pat.matcher(matcher.group(0));
			
			if (ip_matcher.find()) {
				ip = ip_matcher.group(1);
				}

			if (eth_matcher.find()) {
				mac = eth_matcher.group(1);
				}

			if (dbg) System.out.println("ip=" + ip + "; mac=" + mac + "; name=" + name);
			
			if (ip != null && mac != null) {
				NICInfo nicinfo = new NICInfo(name, mac);
				macs.put(ip, nicinfo);
				}
			}
		} 
	finally {
		if ( in != null ) in.close();
		}
	return macs;
    }

public static void main(String[] args) throws Exception
	{
	Args myargs = new Args(args);
	if (myargs.flagExists("debug")) {
		dbg = true;
		}
	System.out.println("[1] Mac Addresses: " + getMAC());	
	
	System.out.println("\n[2] Using java networkinterface methods");	
 	Enumeration<NetworkInterface> infs = NetworkInterface.getNetworkInterfaces();
 	List list = Collections.list(infs);
 	for (int n = 0; n < list.size(); n++) {
 		NetworkInterface inf = (NetworkInterface) list.get(n);
 		System.out.println(
 			"name="+inf.getName()
 			+";ip="+Collections.list(inf.getInetAddresses())
 			+";hw="+inf.getHardwareAddress()
 			+";up="+inf.isUp());
 		}
	}

/** 
Contains information about IP addresses and the associated
interface names and MAC address. More than one IP address
can be associated with the same interface.
**/
public static class NICInfo 
{
String interfaceName;
String mac;
 
NICInfo(String interfaceName, String mac) {
	this.interfaceName = interfaceName;
	this.mac = mac;	
	}
/** 
returns the name of the interface. This name may not
be the same as the name returned by {@link NetworkInterface#getName} 
(especially on windows).
**/
public String getInterfaceName() { return interfaceName; }

/** returns the MAC address of the interface **/
public String getMACAddress() { return mac; }

public String toString() {
	return "[NICInfo: name=" + interfaceName + "; MAC=" + mac + "]"; 
	}
} //~class NICInfo


}  //~class NIC
