// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.io;

import java.util.*;
import java.io.*;
import java.awt.*;
import java.awt.event.*;

import fc.util.*;

/** 
Provides a AWT based console with separate display areas for
input, output and error streams. Useful for interactive sessions
on systems that have AWT enabled. 
<p>
<b>Thread Safety:</b> This class is <u>not</u> MT safe and
higher level synchronization should be used if necessary.

@author hursh jain
**/
public class Console
{
static final boolean dbg = false;

public static final int DEFAULT_MAX_INPUTLINE_LENGTH = 5000;
public static final int DEFAULT_FONTSIZE = 24;

InputPanel 	inPanel;
OutputPanel outPanel;
ErrorPanel 	errPanel;
int fontSize;
int maxLineLength;

/** 
Consructs and shows the console using the title <tt>"Console"</tt>
a default {@link #DEFAULT_FONTSIZE font size}, and a maximum input
line length of {@link #DEFAULT_MAX_INPUTLINE_LENGTH}. Read/write
streams are shown as separate panes within the same window.
**/
public Console() {
	this("Console", 24, DEFAULT_MAX_INPUTLINE_LENGTH, false);
	}

/**
Constructs and shows the console.

@param	fontSize		the size of the display font
@param	seperatePanels	<tt>true</tt> to show the in/out/err panels
						as separate windows, <tt>false</tt> to show
						them within 1 large window.
**/
public Console(int fontsize, boolean seperatePanes) {
	this("Console", fontsize, DEFAULT_MAX_INPUTLINE_LENGTH, seperatePanes);
	}
	
/** 
Constructs and shows the console. 
<p>
Note: After the console is constructed, the read/write streams can
be obtained via the {@link getInputReader}, {@link
#getOutputStream} and {@link #getErrorStream} methods.

@param	title 			the title for the console
@param	fontSize		the size of the display font
@param	maxLineLength	the maximum length of the input line. Lines
						greater than this length will be truncated
						to this length.
@param	seperatePanels	<tt>true</tt> to show the in/out/err panels
						as separate windows, <tt>false</tt> to show
						them within 1 large window.
**/
public Console( String title, int fontSize, int maxLineLength, 
				boolean seperatePanels)
	{
	this.fontSize = fontSize;
	this.maxLineLength = maxLineLength;
	Font font = new Font("Monospaced", Font.PLAIN, fontSize);
	
	inPanel = new InputPanel();
	outPanel = new OutputPanel();
	errPanel = new ErrorPanel();

	if (seperatePanels) 
		{
		Frame inf = new Frame(title + " Input");
		inf.setFont(font);
		inf.add(inPanel); 
		
		Frame outf = new Frame(title + " Output");
		outf.setFont(font);
		outf.add(outPanel); 

		Frame errf = new Frame(title + " Error");
		errf.setFont(font);
		errf.add(errPanel);
 
 		addWindowCloseListener(inf);
 		addWindowCloseListener(outf);
 		addWindowCloseListener(errf);
		
		setSize(errf);  
		setSize(inf); 
		setSize(outf); 
	
 		errf.show();	
		inf.show(); 
		outf.show();
	
		Point p = errf.getLocationOnScreen();
		inf.setLocation(p.x + 40, p.y + 40);		
		outf.setLocation(p.x + 80, p.y + 80);
		}

	else 
		{
		Frame f = new Frame(title);
		f.setFont(font);
		//f.setLocation(new Point(0,0));
		f.setLayout(new GridLayout(3,1));
		f.add(inPanel);
		f.add(outPanel);
		f.add(errPanel);
		setSize(f);
		addWindowCloseListener(f);
		f.show();
		}
	}
	
final void setSize(Frame f) {
	f.setSize(620,460);
	}
	
final void addWindowCloseListener(Frame f)
	{
	f.addWindowListener(new WindowAdapter() {
		public void windowClosing(WindowEvent e) {
			e.getWindow().dispose();
			}
		});
	}	
	
/** 
<b>NOT YET IMPLEMENTED</b>

Exits the application (via <tt>System.exit(0)</tt> when the
console window is closed. If the console uses separate input,
output and error windows, the application will be exited when all
3 windows are closed. 

@param	val		<tt>true</tt> to exit on close. By default, this
				is <tt>false</tt>
**/
public void exitOnClose(boolean val) {
	throw new IllegalArgumentException("Not yet implemented");
	}	

/** 
Returns the AWT panel containing the input section. Can be
used to add event handlers etc. 
**/
public Panel getInputPanel() {
	return inPanel;
	}

/** 
Returns the AWT panel containing the output section. Can be
used to add event handlers etc. 
**/
public Panel getOutputPanel() {
	return outPanel;
	}

/** 
Returns the AWT panel containing the error section. Can be
used to add event handlers etc. 
**/
public Panel getErrorPanel() {
	return errPanel;
	}


/** 
Returns a Reader with the input window as it's source.
**/
public Reader getInputReader() {
	return inPanel.getInputReader();
	}

/** 
Returns a OutputStream with the output window as it's destination 
**/
public OutputStream getOutputStream() {
	return outPanel.out;
	}

/** 
Returns a Writer with the error window as it's destination 
**/
public OutputStream getErrorStream() {
	return errPanel.err;
	}
	
class InputPanel extends Panel
{
TextArea textArea;
TextField textField;
ConsoleInputReader in;
	
public InputPanel()	
	{
	setLayout(new BorderLayout());
	add(makeLabel("Input"), BorderLayout.NORTH);
	textArea = new TextArea();
	textArea.setEditable(false);
	add(textArea, BorderLayout.CENTER);

	Panel textFieldPanel = new Panel();
	textFieldPanel.setLayout(new BorderLayout());
	Label tflabel = new Label("Enter input: " );
	textField = new TextField();
	textFieldPanel.add(tflabel, BorderLayout.WEST);
	textFieldPanel.add(textField, BorderLayout.CENTER);

	add(textFieldPanel, BorderLayout.SOUTH);
	in = new ConsoleInputReader(this);
	}	
	
Reader getInputReader() {
	return in.getInputReader();
	}
}		

class OutputPanel extends Panel
{
TextArea 			textArea;
ConsoleOutputStream out;

public OutputPanel()	{
	setLayout(new BorderLayout());
	add(makeLabel("Output"), BorderLayout.NORTH);
	textArea = new TextArea();
	add(textArea, BorderLayout.CENTER);
	out = new ConsoleOutputStream(textArea);
	}
}		

class ErrorPanel extends Panel
{
TextArea 			 textArea;
ConsoleOutputStream  err;

public ErrorPanel()	{
	setLayout(new BorderLayout());
	add(makeLabel("Error"), BorderLayout.NORTH);
	textArea = new TextArea();
	add(textArea, BorderLayout.CENTER);
	err = new ConsoleOutputStream(textArea);
	}
}		

class ConsoleInputReader 
	{
	String nl = IOUtil.LINE_SEP;
	TextArea  textArea;
	TextField textField;
	Reader	reader;
	Writer  writer;

	public ConsoleInputReader(InputPanel panel) 
		{
		textArea = panel.textArea;
		textField = panel.textField;
		BoundedCharQueue q = new BoundedCharQueue(maxLineLength);
		reader = q.getReader();
		writer = q.getWriter();

		textField.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) 
				{
				String str = textField.getText();
				int strlen =  str.length();
				if (strlen > maxLineLength)
					str = str.substring(0, maxLineLength);
				if (dbg) {
					System.out.println("Input text field: enter key pressed");				
					System.out.println("data: " + str);
					}
				textField.setText("");
				textArea.append(str);
				textArea.append(nl);
				try {
					writer.write(str);
					writer.write(nl);
					}
				catch (Exception ex) {
					ex.printStackTrace();
					}
				}
			});
		} //~init
		
	Reader getInputReader() {
		return reader;
		}
	}

static class ConsoleOutputStream extends OutputStream 
	{
	TextArea text;
	public ConsoleOutputStream (TextArea text) {
		this.text = text;
		}
		
	public void close() {  }
	
	public void flush() { }

	public void write(int b) {
		text.append(String.valueOf((byte)b));
		}

	public void write(byte[] buf, int off, int len) 
		{		
		text.append(new String(buf, 0, len));
		}
	}

Label makeLabel(String str) {
	Label label = new Label(str);
	Font f = new Font("Monospaced", Font.BOLD, fontSize);
	label.setFont(f);
	label.setForeground(Color.WHITE);
	label.setBackground(Color.DARK_GRAY);
	return label;
	}

/** Unit testing: <tt>java fc.io.Console -help</tt> **/
public static void main(String[] args) throws Exception
	{
	Args myargs = new Args(args);
	myargs.setUsage("java fc.io.Console <options>\n" +
		"\n  -help          for this help message" +
		"\n  -title string  title for console windows (default: 'console')" + 
		"\n  -font  size    the font size for the windws (default: "+ DEFAULT_FONTSIZE +")" +
		"\n  -maxin	num     the maximum size of the input line (default: " + DEFAULT_MAX_INPUTLINE_LENGTH + " chars" +
		"\n  -sep           for separate windows (default: combined window)" +
		""
		);
		
	if (myargs.flagExists("help"))
		myargs.showError();	
		
	String title = myargs.get("title", "console");
	int font = Integer.parseInt(myargs.get("font", String.valueOf(DEFAULT_FONTSIZE)));
	int maxin = Integer.parseInt(myargs.get("maxin",String.valueOf(DEFAULT_MAX_INPUTLINE_LENGTH)));
	boolean sep = myargs.flagExists("sep");

	Console c = new Console(title, font, maxin, sep);

	PrintStream out = new PrintStream(c.getOutputStream());
	out.println("hello world");
	out.println("Type some input in the input window and hit return)['exit' to end]");
	
	String str = "";
	
	BufferedReader reader = new BufferedReader(c.getInputReader()); 
	while (! str.equalsIgnoreCase("exit")) {
		str = reader.readLine();
		out.println("You typed: " + str);
		}
	System.exit(0);
	}
}          //~class Console