// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;

/**
Misc collection related utility methods.

Inspired by similar on the web.
*/
public final class C 
{
/**
Create an return a list from the specified args.
*/
public static List list(final Object... args) 
	{
	//Arrays.asList is a fixed sized list backed by Args, cannot add to it
	//return Arrays.asList(args);
	
	//return a proper list instead
	final ArrayList list = new ArrayList(args.length);
	for (int n = 0; n < args.length; n++) {
		list.add(args[n]);
		}
	return list;
	}

/**
Create an return a Set from the specified args.
*/
public static Set set(final Object... args) 
	{
	Set result = new HashSet(args.length);

	result.addAll(Arrays.asList(args));
	return result;
	}

/**
Create an return a Map from the specified args, which are treated as key, value pairs in sequence.
Example:
<code>
Map map = map("a", "hello", "b", 123, "c", new int[] {4,5,6});
</code>
*/

public static Map map(final Object...objects) 
	{
	final Map result = new HashMap(objects.length/2);
	if (objects.length % 2 == 1) {
		throw new IllegalArgumentException("Need (key, value) pairs - number of args should be even!");
	}
	for (int n = 0; n < objects.length; n+=2)
		{
		result.put(objects[n], objects[n+1]);
		}
	return result;
	}

/**
Create an return a Map from the specified args.
Example:
<code>
Map map = map(e("a", "hello"), e("b", 123));
</code>
*/
public static Map map(final Entry... entries) 
	{
	final Map result = new HashMap(entries.length);
	
	for (Entry entry : entries) {
		result.put(entry.key, entry.val);
		}
	return result;
	}

/**
Alias for the e method
*/
public static Entry entry(final Object key, final Object val) 
	{
	return new Entry(key, val);
	}

/**
Create a map entry to pass to the map method. values can be null. The key, value
pairs are accessed directly after they are put in the map.
*/
public static Entry e(final Object key, final Object val) 
	{
	return new Entry(key, val);
	}

private static class Entry 
	{
	Object key;
	Object val;
	
	public Entry(final Object key, final Object val) {
		this.key = key;
		this.val = val;
		}
	}

/** 
Pick NON-duplicate N random items from a list. If N < number of unique items in the list, throws
a IllegalArgumentException. Also throws a IllegalArgumentException if not able to get unique
items after 16 tries (per item). To specify a different retry count (per item), use the variant
of this method that takes the retry parameter.

@param count  number (N) items to choose.
*/
public static Set randomNNoDuplicates(final List list, final int count) throws IllegalArgumentException
	{
	return randomNNoDuplicates(list, count, 16);
	}
				
/**
alias for the {@link #randomNNoDuplicates(List, int)} method
*/
public static Set randomNNoDups(final List list, final int count) throws IllegalArgumentException
	{
	return randomNNoDuplicates(list, count, 16);
	}		
	
/**
Pick NON-duplicate N random items from a list. If N < number of unique items in the list, throws
a IllegalArgumentException. Also throws a IllegalArgumentException if not able to get unique
items after the specified (per item). 

@param count   number (N) items to choose.
@param retries number of retries to get a unique random item before bailing/throwing an Exception
*/
public static Set randomNNoDuplicates(final List list, final int count, final int retries) throws IllegalArgumentException
	{
	if (list.size() < count) {
		throw new IllegalArgumentException("The specified list does not contain [" + count + "] items");
		}
				
	final Set set = new HashSet();
	set.addAll(list);
	
	//these many unique items exist ?
	if (set.size() < count) {
		throw new IllegalArgumentException("The specified list does not contain [" + count + "] unique items");
		}
	set.clear();
	Random r = new Random();
	
	for (int n = 0; n < count; n++) 
		{
		Object item = list.get(r.nextInt(list.size()));
		int tries = 0;
		while (set.contains(item)) 
			{
			item = list.get(r.nextInt(list.size()));
			tries++;
			if (tries > retries) {
				throw new IllegalArgumentException("Too many tries to get a unique item");
				}
			}
		set.add(item);
		}
	return set;
	}

/** unbox and return 0 if null */
public static short unbox(Short s) {
	if (s == null) return 0;
	else return s;
	}

/** unbox and return 0 if null */
public static short unboxShort(Object s) {
	return unbox((Short)s);
	}

/** unbox and return 0 if null */
public static int unbox(Integer i) {
	if (i == null) return 0;
	else return i;
	}

/** unbox and return 0 if null */
public static int unboxInt(Object i) {
	return unbox((Integer)i);
	}
	
/** unbox and return 0 if null */
public static long unbox(Long l) {
	if (l == null) return 0;
	else return l;
	}

/** unbox and return 0 if null */
public static long unboxLong(Object i) {
	return unbox((Long)i);
	}
	
/** unbox and return 0 if null */
public static double unbox(Double d) {
	if (d == null) return 0;
	else return d;
	}

/** unbox and return 0 if null */
public static double unboxDouble(Object d) {
	return unbox((Double)d);
	}
	
/** unbox and return '0' if null */
public static char unbox(Character c) {
	if (c == null) return '0';
	else return c;
	}

/** unbox and return '0' if null */
public static char unboxChar(Object c) {
	return unbox((Character)c);
	}
	
/** unbox and return false if null */
public static boolean unbox(Boolean b) {
	if (b == null) return false;
	else return b;
	}

/** unbox and return false if null */
public static boolean unboxBool(Object b) {
	return unbox((Boolean)b);
	}

/** unbox and return false if null */
public static boolean unboxBoolean(Object b) {
	return unbox((Boolean)b);
	}


/** 
Returns true if the specified collection is null or empty.
**/
public static boolean nullOrEmpty(Collection c) 
	{
	return c != null && c.size() == 0;
	}


public static void main(String args[]) 
	{
	List list = list("a", "b", 1);
	System.out.println(list);
	
	Set set = set("a", "b", "b", "c", 1);
	System.out.println(set);
	
	Map map = map(entry("a", 123), entry("b", new Object()), e(1, "1"), e(2, null));
	System.out.println(map);

	Map map2 = map("a", "hello", "b", 123, "c", new int[] {4,5,6}, "s", null);
	System.out.println(map2);

	//traditional way
	String s = (String) map2.get("s");
	System.out.println("string:" + s);
	
	//unbox util
	int i = unboxInt(map2.get("b"));
	System.out.println("int:" + i);
	
	//any null value can be unboxed at any type
	int i2 = unboxInt(map2.get("s"));
	System.out.println("int:" + i2);

	//any null value can be unboxed at any type
	boolean b = unboxBool(map2.get("s"));
	System.out.println("boolean: " + b);
	
	List list2 = list("a", "b", "a", "c", "d", "c", "g");
	System.out.println(randomNNoDuplicates(list2, 3));

	System.out.println("==Exception expected===");
	try {
		list2 = list("a", "b");
		System.out.println(randomNNoDuplicates(list2, 3));
		}
	catch (Exception e) {
		System.out.println(e);
		}
	System.out.println("==end Exception expected===");

	list2 = list("a", "b", "c");
	System.out.println(randomNNoDuplicates(list2, 3));

	for (int n = 0; n < 300; n++) {
		list2.add("x");
		}

	Collections.shuffle(list2);
	
	System.out.println("==should more than likely throw a retry count exception==");
	try {
		System.out.println(randomNNoDuplicates(list2, 3));
		}
	catch (Exception e) {
		System.out.println(e);
		}

	System.out.println("==should more than likely NOT throw a retry count exception (trying 1000 times per item) ==");
	System.out.println(randomNNoDuplicates(list2, 3, 1000));
	}
}

