// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;


/** A set of char ranges. **/
public class CharRangeSet
{
CharRange seedRange;	
List union;
List intersect;

/**
Constructs a new CharRangeSet with the initial set containing
only the specified charrange

@param	cs	the initial charrange
**/
public CharRangeSet(CharRange cs) 
	{
	Argcheck.notnull(cs);
	seedRange = cs;	
	//do _not_ lazy instantiate, otherwise have to put in
	//null checks in other places
	union = new ArrayList();
	intersect = new ArrayList();
	}

/**
Adds this specified range as a union to the existing set of 
ranges (for purposes of {@link inRange(char)} method). 
Overlapping ranges are ok.

@param	cs	a charrange to unite with
@throws IllegalArgumentException 
			if the specified range was null
**/
public void union(CharRange r) 
	{
	Argcheck.notnull(r);
	union.add(r);
	}

/**
Adds the specified range as an intersection to the existing
ranges (for purposes of {@link inRange(char)} method). 
Overlapping ranges are ok. 

@param	r	the range to add
**/
public void intersection(CharRange r) 
	{
	Argcheck.notnull(r);	
	intersect.add(r);
	}

/** 
Consider a set of ranges <tt>A</tt>, <tt>B</tt> added as a union
(logical <tt>or</tt>) and ranges <tt>C</tt> and <tt>D</tt> added as an
intersection (logical <tt>and</tt>). A character <tt>c</tt> is in range if
it exists in:

<blockquote> <tt>

(A.inRange(c) || B.inRange(c) || ...) && C.inRange(c) && D.inRange(c)
&& ...

</tt></blockquote> 

This can be generalized to an arbitrary number of sub ranges. If
intersection or union ranges don't exist, then they are not
considered in the above expression. Note, the interaction may be
subtle if any of the ranges (<tt>A, B, C...</tt>etc) are
individually negated because in that case the inRange method for
that negated range would return true if the specified character
was <b>not</b> in that range.

@return	<tt>true</tt> if the passed in character is allowed by this 
		set of ranges.

**/
public boolean inRange(char c) 
	{
	boolean result = seedRange.inRange(c);
	for (int i = 0, n = union.size(); i < n; i++) {
		CharRange item = (CharRange) union.get(i);
		result = result || item.inRange(c);
		}		
	for (int i = 0, n = intersect.size(); i < n; i++) {
		CharRange item = (CharRange) intersect.get(i);
		result = result && item.inRange(c);
		}	
	return result;
	}

public String toString() {
	StringBuffer buf = new StringBuffer(128); 
	buf.append("CharRangeSet:[");	
	
	if (intersect.size() > 0)
		buf.append("(");

	buf.append(seedRange);

	int len = union.size();
	if (len > 0) 
   		{
   		buf.append(" || " );
   		for (int i = 0; i < len; i++) {
   			buf.append(union.get(i).toString());
   			if (i < (len-1))
   				buf.append(" || ");
   			}			
		}
		
	len = intersect.size();
	if (len > 0) 
		{
   		buf.append(") && " );			
		for (int i = 0; i < len; i++) {
			buf.append(intersect.get(i).toString());
			if (i < (len-1))
				buf.append(" && ");
			}			
		}
		
	buf.append("]");
	return buf.toString();
	}


public static void main(String[] args)
	{
	CharRange r = new CharRange('b', 'd');
	CharRange r2 = new CharRange('x', 'y');
	CharRangeSet crs = new CharRangeSet(r);
	crs.union(r2);
	System.out.println("constructed: " + crs);
	
	//normal
	test(crs);
	
	crs.intersection(new CharRange('a', 'x'));
	System.out.println("added intersection, now: ");
	System.out.println(crs);
	test(crs);

	crs.union(new CharRange('a', 'z'));
	crs.intersection(new CharRange('s', 't'));
	System.out.println("added union, intersection, now: ");
	System.out.println(crs);
	
	test(crs);	
	}

private static void test(CharRangeSet r) 
	{
	System.out.println("'b' in range:" + r.inRange('b'));
	System.out.println("'z' in range:" + r.inRange('z'));
	System.out.println("'s' in range:" + r.inRange('s'));
	System.out.println("'m' in range:" + r.inRange('m'));
	} 
}