// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import fc.util.*;
import fc.io.*;
import java.io.*;
import java.util.*;
import java.util.logging.*;

/**	
Loads properties from a file. Trims whitespace from
property values, in contrast to ({@link java.util.Property} does not trim whitespace from
values, so "x = y " will return "y ", which is hard to
debug).
<p>
ThreadSafety: This class <b>is</b> thread safe and can be used by multiple 
threads concurrently.

@author hursh jain 
@version 1.1 5/31/2002
*/
public class FilePropertyMgr extends PropertyMgr
{
String myname = getClass().getName();

Properties 	props;
File 		propertyFile;

public FilePropertyMgr(File propertyFile) 
throws IOException
	{
	props = new Properties();
	this.propertyFile = propertyFile;
	try {
		props.load(new BufferedInputStream(
					new FileInputStream(propertyFile)));
		log.info(myname, "; Loaded properties from file: ", propertyFile);
		log.bug(IOUtil.propertiesToString(props));
		}
	catch (IOException e) { 
		log.error("Cannot open file: ", propertyFile.getAbsolutePath());
		log.error("*** PropertyMgr will not be available ***", e);
		throw e;
		}
	}
	
public String get(String name)
	{
	final String prop = props.getProperty(name);
	if (prop != null) {
		return prop.trim();		
		}
	return prop;
	}

public String set(String name, String value)
	{
	Object obj = props.setProperty(name, value);
	return (String) obj;
	}

public void save() throws IOException
	{
	String header = "Properties saved on: " + new Date();
	props.store( new FileOutputStream(propertyFile),  header );
	}


/**
Returns a description but the exact details of said description
are unspecified and subject to change. However, the following
may be regarded as typical:
<tt>
[getClass().getName()] contains the following properties: <br>
[List of properties]
</tt>
**/
public String toString()
	{
	String temp = myname + "; contains the following properties: " + IOUtil.LINE_SEP;
	temp += IOUtil.propertiesToString(props);
	temp += "------------------------";
	return temp;
	}

public static void main(String[] args)
	{
	new Test();
	}

static private class Test 
{
Test() 
{
final String filename = "./FilePropertyMgr_test.txt";
try {

	class FooFilePropertyMgr extends FilePropertyMgr 
		{	
		public FooFilePropertyMgr(String filename) throws Exception {
			super(new File(filename));
			}
		
		protected void handleError(String msg) {
			System.out.println("Error: " + msg);
			}
		}
		
	FilePropertyMgr fp = new FooFilePropertyMgr(filename);
	System.out.println("got required property 'foo' = " + fp.getRequired("foo"));
	System.out.println("got required property 'abc' = " + fp.getRequired("abc"));
	System.out.println("got required property 'x.y' = " + fp.getRequired("x.y"));
	System.out.println("got required property 'num' = " + fp.getRequiredInt("num"));
	System.out.println("got required property 'bool' = " + fp.getRequiredBoolean("bool"));

	System.out.println("got property 'num2' = " + fp.getInt("num2", 123));
	System.out.println("got property 'bool2' = " + fp.getBoolean("bool2", false));

	System.out.println("Getting bad format properties...");
	System.out.println("got required property 'num3' = " + fp.getRequiredInt("num3"));
	System.out.println("got required property 'bool3' = " + fp.getRequiredBoolean("bool3"));
	System.out.println("got property 'num3' = " + fp.getInt("num3", 123));
	System.out.println("got property 'bool3' = " + fp.getBoolean("bool3", false));

	System.out.println("now calling the FilePropertyMgr.toString()");
	System.out.println(fp.toString());
	
	System.out.println("The following should exit the test");
	fp = new FilePropertyMgr(new File(filename));
	System.out.println("got property 'test.don'texist' = " + fp.getRequired("test.don'texist"));
	} 
catch (Exception e) {
 	e.printStackTrace();
 	}	
}

} //~end Test


}           //~class FilePropertyMgr