// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.io.*;
import java.util.*;
import fc.io.*;
import fc.util.*;

import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;
import javax.imageio.*;

/* Misc. image related utils */
public final class ImageUtil
{

/**
Resizes the specified image file to the new width and height and writes it to
the targetFile (creating the targetFile if it does not already exist).
<p>
The format string is the output format of the file, like "jpg", "gif" etc. This
can be different than the format of the source file. The understood formats
are those understood {@link javax.io.ImageWriter} (the usual suspects like gif,
jpg, png, etc, all seem to work).
*/
public static void resize(File sourceFile, File targetFile, String format, int newWidth, int newHeight)
throws IOException	
	{
	BufferedImage resized = resize(sourceFile, newWidth, newHeight);
	ImageIO.write(resized, format, targetFile);
	}


/**
Resizes the specified image file to the new width and height and writes it to
the targetFile (creating the targetFile if it does not already exist).
<p>
The output format is the same as the input format and is intuited from the file
name extension of the specified source file (therefore, source files specified
to this method, should be like <code>foo.gif</code>, <code>foo.jpg</code>, etc).
*/
public static void resize(File sourceFile, File targetFile, int newWidth, int newHeight)
throws IOException	
	{
	String name = sourceFile.getName();	 
	int pos = name.lastIndexOf(".");
    if (pos == -1) {
    	throw new IOException("The specified source file: [" + sourceFile.getAbsolutePath() + "] has no extension, and this method needs an extension (like .gif, .jpg etc)");
    	}
	String extension = name.substring(pos+1, name.length()); 
	//System.out.println(extension);
	resize(sourceFile, targetFile, extension, newWidth, newHeight);
	}

/**
Resizes the specified image file to the new width and height and returns the
new image.
*/
public static BufferedImage resize(File sourceFile, int newWidth, int newHeight)
throws IOException	
	{
	BufferedImage sourceImage = ImageIO.read(sourceFile);
	//WritableRaster raster = image.getRaster(); //for pixel manipulation, not needed

    int srcWidth = sourceImage.getWidth();
    int srcHeight = sourceImage.getHeight();
    
    BufferedImage targetImage = new BufferedImage(newWidth, newHeight, BufferedImage.TYPE_INT_RGB);
	Graphics2D g2 = targetImage.createGraphics();
	AffineTransform transform = AffineTransform.getScaleInstance((double)newWidth/srcWidth, (double)newHeight/srcHeight);

	g2.setRenderingHint(RenderingHints.KEY_INTERPOLATION,
						RenderingHints.VALUE_INTERPOLATION_BICUBIC);
						
	g2.drawImage(sourceImage, transform, null);
	g2.dispose();	

	return targetImage;
	}

public static void main (String args[]) throws IOException	
	{
	Args myargs = new Args(args);
	myargs.setUsage(myargs.getMainClassName() + " -source path-to-source-image -target path-to-write-target-img -width new-width -height new-height");
	
	File source = new File(myargs.getRequired("source"));
	File target = new File(myargs.getRequired("target"));
	int width = myargs.getRequiredInt("width");
	int height = myargs.getRequiredInt("height");
	
	if (! source.exists()) {
		System.out.println("Cannot read source file: " + source.getAbsolutePath());
		System.exit(1);
		}

	resize(source, target, width, height);	
	}
}
