// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

/**
Allows measuring the start and stop time and (hence the elapsed time) of
some event. ("event" meaning something of interest, for example, a method
call). Can be started/stopped repeatedly. (cumulative time across all such
start/stops are available via the {@link #cumulativeTime} method).
<p>
All times are in <i>nano-seconds</i>. 
<p>
Thread Safety: This class is <b>not</b> threadsafe and it's method
do <b>not</b> acquire any locks to reduce any time skew due to
lock acquisition. Multiple threads should use separate NanoWatch
objects or alternatively, higher level synchronization.

@author 	hursh jain
@see		Watch	A lower precision (milliseconds) watch.
*/
public final class NanoWatch extends Watch
{
public NanoWatch(String name) {
	super(name);
	}

public NanoWatch() {
	super("DefaultNanoWatch/" + Thread.currentThread().getName());
	}

/** 
Start measuring time. Call this method just before calling the
method/code to be instrumented. This method does <b>not</b> reset the
Watch, so the reset method should be called before calling this method
<b>again</b>.
*/
public NanoWatch start() {
	startTime = System.nanoTime();	
	running = true;
	return this;
	}
	
/** Stop measuring the time */
public void stop() {
	stopTime = System.nanoTime();
	cumulativeTime += (stopTime - startTime);
	running = false;
	}


/** 
Returns the time elapsed since the Watch was started. If the watch was
started and stopped, returns the time between the start/stop interval.

@throws RuntimeException	if the watch was never started before calling
							this method.
*/
public long time() 
	{
	if (startTime == -1)
		throw new RuntimeException("You need to start the watch at least once before calling this method");

	if (running)
		return System.nanoTime() - startTime;
	
	return	stopTime - startTime;
	}

/** 
Returns the time elapsed since the Watch was started. (this method is
an alias for {@link #time} method).
*/
public long getTime() 
	{
	return time();
	}

private static final double ms_factor = 1.0 / 1000000.00D;

/** 
Returns the elapsed time in milliseconds. Useful when nanosecond values are
too hard to read.
*/
public double getTimeInMillis() 
	{
	long t = time();  
	return t * ms_factor;
	}


/** 
Returns the total time recorded by this Watch (across several starts/stops)
*/
public long cumulativeTime() 
	{
	if (! running) 
		return cumulativeTime;
	
	return cumulativeTime +	(System.nanoTime() - startTime);
	}

/** 
Reset all values to zero. This method should be called
before this object is used <b>again</b>.
*/
public void reset() 
	{
	startTime = stopTime =  0;
	}

/** Is the Watch currently running ? */
public boolean isRunning() {
	return running;
	}

/** 
Get the start time (in nanoseconds), the start time is arbitrary (see 
{@link System#nanoTime}).
*/
protected long getStart() {
	return startTime;
	}

/** 
Get the stop time (in nanoseconds). Useful only as a difference from
the start time
*/
protected long getStop() {
	return stopTime;
	}

public String toString() 
	{
	java.text.NumberFormat nf = java.text.NumberFormat.getNumberInstance();
	nf.setGroupingUsed(true);
	
	String str = myname;

	str += ": Cum.Time=[" + nf.format(cumulativeTime()) + " nanoseconds]";
	
	if (running) 
		str += "; Elapsed=[" + nf.format(time()) + " nanoseconds]";
		
	return str;
	}

public static void main(String[] args)
	{
	fc.util.Watch t1 = new fc.util.NanoWatch("NanoWatch 1");
	t1.start();
	
	new Thread(new Runnable() { 
		public void run() {
			try { 
				Watch t2 = new NanoWatch();
				t2.start(); Thread.currentThread().sleep(20); t2.stop(); 
				System.out.println("t2.toString():" + t2);
				} 
			catch (Exception e) { e.printStackTrace(); }
			} 
		}).start();
	
	//following should return -1
	System.out.println("NanoWatch 1, total time taken:" + t1.time());

	System.out.println("NanoWatch 1, time=" + t1.time());
	System.out.println("NanoWatch 1, before-being-stopped, toString():" + t1);
	t1.stop();
	System.out.println("NanoWatch 1, is running ? " + t1.isRunning() );
	System.out.println("NanoWatch 1, after-being-stopped, toString():" + t1);
	System.out.println("NanoWatch 1, elapsed time:" + t1.time());
	System.out.println("NanoWatch 1, cumulative time taken:" + t1.cumulativeTime());
	System.out.println("NanoWatch 1, elapsed time:" + t1.time());
	}

} //~class NanoWatch
