// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;

/** 
A range of integral values.
<p>
Note, many methods in this particular class have the same
signatures as in the apache <tt>commons.lang</tt> package but the
<tt>implementation</tt> was strictly clean-room.
<p>
Thread Safety: This class is not thread safe in the face of any
concurrent modification (via changing the start/end numbers or
union/intersection operations).

@author hursh jain
**/
public class NumberRange extends Range
{
long start;
long end;

/** 
Constructs a number range between the specified
start and end numbers (both inclusive). 

@throws IllegalArgumentException 
			if the end number is lesser (via a <tt>&lt;</tt> comparison) 
			than the start number (equal to the start number is ok)
**/
public NumberRange(long start, long end) 
	{
	Argcheck.istrue(start <= end, "end number lesser than start number in the specified range, start=["+start+"], end=["+end+"]");
	this.start = start;
	this.end = end;
	}
	
/** Get the start number for this number range **/
public long getStart() 
	{
	return start;
	}

/** 
Get the end number for this number range. 
**/
public long getEnd() 
	{
	return end;
	}


/** Is the passed in number inside this range **/
public boolean inRange(long c) 
	{
	boolean result = (c >= start) && (c	<= end);
	
	if (negated)
	 	return !result;
	else
		return result;
	}
 
/** 
Sets the end number for this number range 

@param	c	the end number
@throws IllegalArgumentException 
			if the end number is lesser (via a <tt>&lt;</tt> comparison) 
			than the current start number (equal to the start number is 
			ok)
**/
public void setEnd(long c) 
	{
	Argcheck.istrue(start <= c, "end number lesser than start number in the specified range, start=["+start+"], specified end=["+c+"]");
	end = c;
	}
 
/** Set the start number for this number range 

@throws IllegalArgumentException 
			if the start number is greater (via a <tt>&gt;</tt> comparison) 
			than the current end number (equal to the end number is 
			ok)
**/
public void setStart(long c) 
	{
	Argcheck.istrue(c <= end, "start number greater than end number in the specified range, end=["+end+"], specified start=["+c+"]");
	start = c;
	}

/** Returns the size of this range which is calculated as the
<tt>(endnumber - startnumber) + 1</tt> - i.e., the number of
slots in this range, start and end inclusive.
**/
public long getSize() {
	return (end - start) + 1;
	}
	
	
/** Output a string representation of the number range **/
public java.lang.String toString() 
	{
	String str = "NumberRange:[";
	if (isNegated())
		str +=	"^";
	
	if (start >= 0) 
		str += start;
	else 
		str += "(" + start + ")";
	
	str += "<->";
	
	if  (end >= 0) 
		str += end ;
	else 
		str += "(" + end + ")";
		 
	str += "]";
	return str;
	}
	
public boolean equals(Object obj) 
	{
	if (! (obj instanceof NumberRange))
		return false;
	NumberRange other = (NumberRange) obj;
	return 	(this.start == other.start && 
			this.end == other.end);
	}

public int hashCode() {
	return (int) getSize();
	}


public static void main(String[] args)
	{
	NumberRange r = new NumberRange(-10, 10);
	
	System.out.println("constructed range:" + r);
	System.out.println("Range size=" + r.getSize());

	NumberRange r2 = new NumberRange(-10, 10);
	System.out.println("Constructed range2=" + r2);
	NumberRange r3 = new NumberRange(-10, 9);
	System.out.println("Constructed range3=" + r3);
	
	System.out.println("");
	System.out.println("r.equals(r2)="+r.equals(r2));
	System.out.println("r.equals(r3)="+r.equals(r3));

	System.out.println("r.hashCode="+r.hashCode());
	System.out.println("r2.hashCode="+r2.hashCode());
	System.out.println("r3.hashCode="+r3.hashCode());

	
	//normal
	test(r);
	
	//negate
	r.setNegated(true);
	System.out.println("setting the range to be negated:" + r);
	test(r);

	System.out.println("the following should throw an exception");
	r = new NumberRange(5, 0);
	}

private static void test(NumberRange r) 
	{
	System.out.println("0 in range:" + r.inRange(0));
	System.out.println("1 in range:" + r.inRange(1));
	System.out.println("10 in range:" + r.inRange(10));
	System.out.println("11 in range:" + r.inRange(11));
	System.out.println("-1 in range:" + r.inRange(-1));
	System.out.println("-10 in range:" + r.inRange(-10));
	System.out.println("-11 in range:" + r.inRange(-11));
	}

}          //~class NumberRange