// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.io.*;
import fc.io.*;

/**	
Implementations help load and manage properties from various
sources including java property files, databases, custom
file formats etc. Regardless of where the properties are
loaded from, each property consists of a pair of <b>name</b>
and <b>value</b> strings.

@author hursh jain 
@version 1.0 12/30/2001
*/
public abstract class PropertyMgr
{
protected String 	usage;
protected Log 		log;

/**
Constructs a new PropertyMgr. 
**/
public PropertyMgr() {
	this.log = Log.getDefault();	
	}

/**
Returns the property associated with the specified key or <tt>null</tt>
if the property was not found.
@param 	name	the property key
**/
public abstract String get(String name);


/**
Returns the property associated with the specified key.
@param 	name	the property key
@param 	backup	value to return if the property for the specified key is not found	

@return	value of specified key or backup string
**/
public String get(String name, String backup)
	{
	String val = get(name);
	if (val == null) 
		{
		return backup;	
		}
	else return val;
	}

/**
Returns the property associated with the specified key as a
boolean value. If the property <b>is</b> present, then the
corresponding boolean value is <tt>true</tt> if the property
value is any of the following (case-insensitive):
<blockquote>
<code>
yes, 1, true
</code>
<blockquote>
else <tt>false</tt> is returned.

@param 	name	the property key
@param 	backup	value to return if the property for the 
				specified property is <b>not</b> present	

@return	value of specified key or backup string
*/
public boolean getBoolean(String name, boolean backup) 
	{
    String val = get(name);
    if (val == null)
    	 return backup;
    else
  		 return toBoolean(val);
    }

/**
Returns the property associated with the specified key as a
integer value. If the property is present but cannot be
converted into an integer (via a {@link
Integer.parseInt(String)} call), the backup value will be
returned.

@param 	name	the property key
@param 	backup	value to return if the property for the 
				specified property is <b>not</b> present	

@return	value of specified key or backup string
*/
public int getInt(String name, int backup) 
	{
    String val = get(name);
    if (val != null)
    	{
   		try {
	    	return Integer.parseInt(val);
	    	}
	    catch (NumberFormatException e) {
	    	log.warn("Cannot convert property '" + name + "', into a number, returning backup value " + backup);
	    	}
	    }
	return backup;
    }

/** 
Returns the value corresponding to the specified key. If the
property <b>value</b> is not found, the {@link #handleError} method
is called, which by default prints a stack trace and exits
the application.
<p>
If the handleError method is overriden to <b>not</b> exit
the application, then this method will return <tt>null</tt>
if the specified key is not found.

@param 	name	the property key
**/
public String getRequired(String name)
	{
	String val = get(name);
	if (val == null || val.trim().length() == 0) {
		try { 
			handleError(name);
			}
		catch (Exception e) { 
			/* 
			If the handler throws us an Exception, we pass
			it along as a Runtime Exception (since this
			method does not throw a checked exception). Of
			course, handlers are free to deal with
			Exceptions in any other way they see fit.
			*/
			throw new RuntimeException(e); 
			}
		}
	return val;
	}

/*
Returns the value obtained via {@link getRequired(String)}
as as a boolean. The boolean value returned is
<code>true</code> if the property value is any of the
following (case-insensitive):
<blockquote>
<code>
yes, 1, true
</code>
<blockquote>
else <tt>false</tt> is returned.
*/
public boolean getRequiredBoolean(String name) 
	{
	String val = getRequired(name);
	return toBoolean(val);	
	}

/*
Returns the value obtained via {@link getRequired(String)} as
as an integer. If the property cannot be converted into an integer (via a {@link
Integer.parseInt(String)} call), the resulting action would
be the same as if the required key was not present.
*/
public int getRequiredInt(String name)
	{
	String val = getRequired(name);
   	
   	int result = 0;
   	
   	try {
	    result = Integer.parseInt(val);
	    }
	catch (NumberFormatException ne) {
	    log.warn("Cannot convert property '" + name + "', into a number");
		try { 
			handleError(name);
			}
		catch (Exception e) { 
			/* 
			If the handler throws us an Exception, we pass
			it along as a Runtime Exception (since this
			method does not throw a checked exception). Of
			course, handlers are free to deal with
			Exceptions in any other way they see fit.
			*/
			throw new RuntimeException(e); 
			}
		}
	
	return result;
	}
	
/**
Sets the property associated with the specified key.
@param 	name	the property key
@param 	backup	the property value	
@return	the previous value of the specified key or null if it did not 
		have one.
**/
public abstract String set(String name, String backup);

/** 
Saves any properties that were set previously. This method does not
need to be called if properties were only read, however if any
property was modified or added, it is essential to call this method
to save any such changes.
**/
public abstract void save() throws IOException;

/** 
Specify program usage information to be output when an error occurs. This 
information should contain a short description of required keys and any
other information that the application expects.
@param str	Usage information
**/
public void setUsage(String str)
	{
	this.usage = str;		
	}

/** 
Default implementation:<br>
Handles all error situation - such as when a required property is
not found. By default, prints an error message to <tt>System.err</tt> and 
exits the JVM by calling <tt>System.exit(1)</tt>. 
This method can be overriden in subclasses to handle errors differently.
@param 		str			the error message
@throws 	Exception	sub classes can throw an Exception, if necessary
**/
protected void handleError(String str) throws PropertyNotFoundException
	{
	log.error("Application is missing required Property: " + str);
	new Exception().printStackTrace(System.err);
	if (usage != null) {
		log.error(usage);
		}
	System.exit(1);
	}

final boolean toBoolean(String val)
	{
	val = val.trim().toLowerCase().intern();
	if ( val == "true" || val == "yes" || val == "1" )
		return true;
	return false;
	}

}	//~class PropertyMgr      	