// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;

/** 
Implements a {@link java.util.Collection} that keeps it's elements in
Sorted order. This class is internally backed by a LinkedList, therefore
iteration is strongly preferred (and much faster) than random access of
it's elements. This class has no equivalent in JDK 1.4 and should be
replaced by a future JDK equivalent if available. Currently java.util
only provides a sorted Set not a sorted Collection (like this one) that
allows duplicate elements.
<p>
<b>Note that this implementation is not synchronized.</b> If multiple
threads access a list concurrently, and at least one of the threads
modifies the list structurally, it <i>must</i> be synchronized
externally.  (A structural modification is any operation that adds or
deletes one or more elements; merely setting the value of an element is not
a structural modification.)  This is typically accomplished by
synchronizing on some object that naturally encapsulates the list.  If no
such object exists, the list should be "wrapped" using the
Collections.synchronizedList method.  This is best done at creation time,
to prevent accidental unsynchronized access to the list: 
 <pre>
    List list = Collections.synchronizedList(new LinkedList(...));
 </pre>

@author hursh jain
**/
public class SortedCollection extends AbstractCollection
{
final boolean dbg = true;
List items;				//interal storage of contained items
Comparator comparator;

/**
Constructs a new, empty SortedCollection, sorted according to the keys' natural
order. All keys inserted into the map must implement the Comparable interface.
Furthermore, all such keys must be mutually comparable: k1.compareTo(k2) must
not throw a ClassCastException for any elements k1 and k2 in the map. If the
user attempts to put a key into the map that violates this constraint, methods
that add elements will throw a <tt>ClassCastException</tt>.
**/
public SortedCollection() {
	this((Comparator)null);
 	}

/**
Constructs a sorted list containing the elements of the specified
collection. The elements are sorted according to the elements' natural 
order

@param  c the collection whose elements are to be placed into this list.
@throws NullPointerException if the specified collection is null.
**/
public SortedCollection(Collection c) {
	this((Comparator)null);
	addAll(c);
	}

/**
Constructs a new, empty SortedCollection, sorted according to the given comparator.
All keys inserted into the map must be mutually comparable by the given
comparator: comparator.compare(k1, k2) must not throw a ClassCastException for
any keys k1 and k2 in the map. If the user attempts to put a key into the map
that violates this constraint, methods that add elements will throw a
<tt>ClassCastException</tt>.

@param c	the comparator that will be used to sort this map. A null value
			indicates that the keys' natural ordering should be used.
**/
public SortedCollection(Comparator c) 
	{
	comparator = c;
	items = new LinkedList();
 	}


public boolean add(Object obj) 
	{
	if (dbg) System.out.println("SortedCollection.add(" + obj + "): ENTER");
	boolean result = false;
	if (items.size() == 0) {
		//add the initial item
		items.add(obj);
		result = true;
		}	
	else {
		 for (ListIterator it = items.listIterator(); it.hasNext(); /*empty*/) 
			{
			Object item = it.next();
			int c = compare(item, obj);
			if (c >= 0) {
				int index = it.previousIndex();
				items.add( (index > 0) ? index : 0, obj); 
				result = true;
				break;
				}
			}
		}
	if (dbg) System.out.println("SortedCollection.add(" + obj + "): EXIT; result=" + result + "; collection=" + items);
	return result;
	}

public Iterator iterator() {
	return items.iterator();
	}
	
public int size() {
	return items.size();
	}	

public String toString()  {
	return items.toString();
	}

/**
Compares two keys using the correct comparison method for this SortedCollection. That
is to say, if a {@link java.util.Comparator} is set for this class, then it
will be used for the comparison, otherwise the specified objects will be
compared via a cast to {@link java.lang.Comparable}
**/
private int compare(Object k1, Object k2) {
    int result = (comparator==null ? ((Comparable)k1).compareTo(k2) : 
    								 comparator.compare(k1, k2));
	if (dbg) System.out.println("SortedCollection.compare(" + k1 + "," + k2 + "): result = " + result);
	return result;
	}

	
public static void main(String[] args) throws Exception
	{
	//AppMgr.setApp(new UnitTestingApp());
	List list = Arrays.asList(new String [] {"x", "c", "d", "a"}); 
	SortedCollection sc = new SortedCollection(list);
	sc.add("z");	
	sc.add("x");	
	sc.add("b");
	sc.add("c");
	System.out.println(sc);	
	}	
	
}          //~class SortedCollection