// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import fc.io.*;
import fc.util.*;

import java.util.*;
import java.lang.reflect.*;

/**  
Makes creating <tt>toString</tt> methods easier. (For example, 
provides ability to introspect and write field values). Idea
inspired by a similar apache/jakarta utility.
<p>
Methods of the form append(String, type) imply the name 
specified by the string (typically a field name) is shown
with value specified by type.
<p>
Example usage:<br>
<tt>foo</tt> and <tt>bar</tt> are fields of this object.
<blockquote>
<pre>
public String toString() {
	return new ToString(this).
		<font color="blue">append</font>("foo","some-value").
		<font color="blue">append</font>("bar",123).
		<font color="blue">render</font>();
	}
</pre>
</blockquote>
Another example:<br>
Automatically prints this entire object using reflection.
<blockquote>
<pre>
public String toString() {
	return new ToString(this).<font color="blue">reflect</font>().
	<font color="blue">render</font>();
	}
</pre>
</blockquote>
<i><b>Note</b>: Don't forget the <tt>render()</tt> call at the end.</i>
<p>
The class only needs to be instantiated once so here's a more
efficient approach:
<blockquote>
<pre>
{ //instance initializer
ToString tostr = new ToString(this);
}
public String toString() {
	return tostr.<font color="blue">reflect</font>().
	<font color="blue">render</font>();
	}
</pre>
</blockquote>
<p>

@author hursh jain
**/
public class ToString 
{
private static	final	boolean dbg 		 = false;
private static 			Style 	defaultStyle = new Style();

Object 			client;
Style			style;
StringBuffer 	result;
boolean			firstFieldDone;

/**  
Creates a ToString for the specified object, using the
default {@link ToString.Style}.
**/
public ToString(Object obj) {
	this(obj, (Style)null);
	}

/**  
Creates a ToString for the specified object using the
specified style
@param 	obj		the target object
@param	style	the formatting style
**/
public ToString(Object obj, Style style) 
	{
	Argcheck.notnull(obj, "target object cannot be null");
	client = obj;
	
	if (style == null)
		this.style = defaultStyle;
	else
		this.style = style;
		
	result = new StringBuffer();
	}

/**  
Creates a ToString for the specified object with the
specified visibility level.

@param 	obj		the target object
@param	style	the formatting style
**/
public ToString(Object obj, Style.VisibleLevel level) 
	{
	Argcheck.notnull(obj, "target object cannot be null");
	client = obj;
	this.style = defaultStyle;
	this.style.reflectVisibleLevel = level;
	result = new StringBuffer();
	}


/** 
Returns the default style object. Changes to this will affect
the default formatting
**/	
public static ToString.Style getDefaultStyle() 
	{
    return defaultStyle;
    }
   
/** 
Sets the style object to use as the default. This style will
be used by default by all new instances of ToString.

@param	 style	the default style
**/
public static void setDefaultStyle(ToString.Style style) 
	{
	Argcheck.notnull(style, "style cannot be null");
	defaultStyle = style;
    }	

/** 
Returns the style being currently used by this instance. 
Modifications to this style this will affect rendering
output appropriately.
**/
public ToString.Style getStyle() {
	return style;
	}
	

/** Returns the internal buffer used to create the string **/
public StringBuffer getBuffer() {
	return result;
	}

/** 
Uses reflection to get the contents of the object. Reflection
does not expand and print all the array values even if the
current style's {@link ToString.Style#expandArray expandArray}
is set to true. To print all array values, use the
<tt>append</tt> methods.
**/
public ToString reflect() 
	{
	try {
		reflectImpl(client, client.getClass());
		}
	catch (IllegalAccessException e) {
		result.append("Cannot convert to string using reflection");
		result.append(e.toString());
		}
	return this;
	}
	
void reflectImpl(Object obj, Class clazz) 
throws IllegalAccessException
	{
	if (dbg) System.out.println("reflectImpl("+clazz+")");
	
	Field[] fields = clazz.getDeclaredFields();
	
	if (dbg) System.out.println("got declared fields: " + Arrays.asList(fields));
	
	//need this call, otherwise getting the fields does not work
	Field.setAccessible(fields, true);
	
	for(int n = 0; n < fields.length; n++) 
		{
		//if (dbg) System.out.println("declaredField["+n+"]; '"+fields[n].getName()+"'="+fields[n].get(obj)); 	
		
		Field f = fields[n];
		int mod = f.getModifiers();

		if (dbg) System.out.println(f.getName() +" modifier="+mod);

		if (! style.reflectStatics && Modifier.isStatic(mod)) {
			if (dbg) System.out.println("reflectStatics = false, ignoring static field: " + f.getName());
			continue;
			}
		
		boolean defaultVis = ! (Modifier.isPublic(mod) 
				|| Modifier.isProtected(mod) 
				|| Modifier.isPrivate(mod));
					

		if (style.ignoredFieldNames.contains(f.getName().toLowerCase())) {
			if (dbg) System.out.println("ignoring: " + f.getName() + " [in ingorelist]");			
			continue;
			}

		if (style.reflectVisibleLevel==Style.VisibleLevel.PUBLIC)
			{
			if (Modifier.isPublic(mod)) { 
				append(f.getName(), f.get(obj));
				}	
			}
		else if (style.reflectVisibleLevel==Style.VisibleLevel.PROTECTED)
			{
			if (Modifier.isPublic(mod) || Modifier.isProtected(mod)) {
				append(f.getName(), f.get(obj));
				}			
			}
		else if (style.reflectVisibleLevel==Style.VisibleLevel.DEFAULT)
			{
			if (Modifier.isPublic(mod) || Modifier.isProtected(mod)
					|| defaultVis) {
				append(f.getName(), f.get(obj));
				}			
			}			
	 	else if (style.reflectVisibleLevel==Style.VisibleLevel.PRIVATE)
			{
			if (Modifier.isPublic(mod) || Modifier.isProtected(mod)
					|| Modifier.isPrivate(mod) || defaultVis) {
				append(f.getName(), f.get(obj));
				}			
			}
		else
			{
			System.out.println("ERROR in ToString().reflecImpl(), this should not happen, toString won't be accurate");
			}
			
		if (dbg) System.out.println("buf=>>"+result+"<<");
		} //~for 

	if (style.reflectSuperClass) {
		Class superclazz = clazz.getSuperclass();
		if (superclazz != null && superclazz != Object.class)
			reflectImpl(obj, superclazz);
		}
	
	}	//reflectImpl


void doStart(StringBuffer buf)
	{
	buf.append(style.startString);

	Class clazz = client.getClass();
	String name = clazz.getName();
	
	if (style.className)
		{
		if (style.fullClassName) {
			buf.append(name);
			}
		else {
			int last = name.lastIndexOf(".");
			last = (last < 0) ? 0 : last + 1; //+1 to skip '.' itself
			buf.append( name.substring(last, name.length()) );
			}
		}
		
	if (style.idHashCode) {
		buf.append("@");
		buf.append(System.identityHashCode(client));
		}	

	buf.append(style.startContent);
	}

void doEnd(StringBuffer buf) {
	buf.append(style.endContent);
	buf.append(style.endString); 
	}

/** Renders the string **/
public String render() 
	{
	StringBuffer finalResult = new StringBuffer(result.length() + 128);
	doStart(finalResult);
	finalResult.append(result);
	doEnd(finalResult); 
	return finalResult.toString();
	}

/** 
Returns information about the current state of the ToString
object itself. To get the target object's string, use the
{@link render} method.
**/
public String toString() 
	{
	return getClass().getName() + "; using: " + style;
	}
	
/** Unit test **/
public static void main(String[] args)
	{	
	System.out.println("=== Test using reflection ===");
	
	Style style = null;
	
	style = new Style();
	style.ignoreFieldName("style");
	style.reflectVisibleLevel = Style.VisibleLevel.PUBLIC;
	style.reflectStatics = true;
	System.out.println("public fields [including statics]");
	System.out.println(new TestClass(style, true));
	System.out.println("");
		
	style = new Style(); 
	style.ignoreFieldName("style");
	style.reflectVisibleLevel = Style.VisibleLevel.PUBLIC;
	System.out.println("public fields only");
	System.out.println(new TestClass(style, true));
	System.out.println("");

	style = new Style(); 
	style.ignoreFieldName("style");
	style.reflectVisibleLevel = Style.VisibleLevel.PUBLIC;
	style.idHashCode = false;
	style.className = false;
	System.out.println("public fields only, no id ref or class name");
	System.out.println(new TestClass(style, true));
	System.out.println("");
	
	style = new Style();
	style.ignoreFieldName("style");
	style.reflectVisibleLevel = Style.VisibleLevel.PROTECTED;
	System.out.println("protected and higher");
	System.out.println(new TestClass(style, true));
	System.out.println("");
	
	style = new Style();
	style.ignoreFieldName("style");
	style.reflectStatics = true;
	style.reflectVisibleLevel = Style.VisibleLevel.DEFAULT;
	System.out.println("package and higher [including statics]");
	System.out.println(new TestClass(style, true));
	System.out.println("");

	style = new Style();
	style.ignoreFieldName("style");
	style.reflectVisibleLevel = Style.VisibleLevel.DEFAULT;
	style.expandArrays = true;
	System.out.println("package and higher fields, arrays EXPANDED");
	System.out.println(new TestClass(style, true));
	System.out.println("");
	
	style = new Style();
	style.ignoreFieldName("style");
	style.reflectVisibleLevel = Style.VisibleLevel.PRIVATE;
	System.out.println("private (all) fields");
	System.out.println(new TestClass(style, true));
	System.out.println("");
	
	style = new Style();	
	style.ignoreFieldName("style");
	System.out.println("default style output");
	System.out.println(new TestClass(style, true));
	System.out.println("");
	
	System.out.println("==== Test without reflection ====");
	style = new Style();	
	System.out.println(new TestClass(style, false));
	System.out.println("");
	
	System.out.println("With no field names");
	style = new Style();	
	style.fieldName = false;
	System.out.println(new TestClass(style, false));
	System.out.println("");
	
	style = new Style();	
	style.reflectVisibleLevel = Style.VisibleLevel.DEFAULT;
	style.expandArrays = true;
	System.out.println("Expanded arrays");
	System.out.println(new TestClass(style, false));
	System.out.println("");
	}

private static class TestClass 
{
private Style 	style;
private boolean useReflection;

TestClass(Style style, boolean useReflection) { 
	this.style = style; 
	this.useReflection = useReflection;
	}

static int staticInt = 10;
public 	static String staticString = "staticString";
			
public 		String   pubString 		= "publicString";
protected 	String   protectedString = "protectedString";
			String   defString 		= "defaultString";
			int[]    intArray		= new int[] { 1, 2, 3};
			double[] doubleArray	= new double[] {1.3, 2.6, 3.9};
			Object[] objectArray 	= new Object[] { null, new Object() };
			List	 someList		= new ArrayList();
private 	String 	 privateString 	= "privateStrng";				

public String toString() 
	{
	if (useReflection)
		return new ToString(this, style).reflect().render();
	else {
		return new ToString(this, style).
			append("intArray", intArray).
			append("doubleArray", doubleArray).
			append("pubString", pubString).render();
		}
	}
} //~TestClass


/** 
Drives the formatting behavior. Behavior different than 
the defaults can be achieved by instantiating a new object
and setting it's properties appropriately.
**/
public static class Style 
	{	
	private List ignoredFieldNames = new ArrayList();

	/**
	Case insensitive field names that will be ignored (for example a
	public field "foo" may be printed otherwise, but if added to this
	list, it would be ignored). Use this method to add as many
	field names as needed.
	**/
	public void ignoreFieldName(String name) {
		ignoredFieldNames.add(name.toLowerCase());
		}
	
	/**
	 The start of the entire string. default to empty: <tt>""</tt> 
	**/
	public String startString = 		"";

	/** 
	The end of the entire string. default to empty: <tt>""</tt> 
	**/
	public String endString = 			""				;

	/** 
	The start of the string <b>after</b> the object classname and
	identity reference. default: <tt>[</tt> 
	**/
	public String startContent = 		"["				;

	/** 
	The end of the string <b>after</b> the object classname and
	identity reference. default: <tt>]</tt> 
	**/
	public String endContent = 			"]";

	/** default: <tt>=</tt> **/
	public String fieldAndValSep = 		"=";

	/** default: <tt>,</tt> **/
	public String fieldSep = 			", ";

	/** default: <tt>{</tt> **/
	public String startArray = 			"{";

	/** default: <tt>}</tt> **/
	public String endArray = 			"}";

	/** default: <tt>,</tt> **/
	public String arrayValSep = 		",";
	
	/** 
	Expand array values, default: <tt>false</tt>. <bNote:</b> 
	expansion only works when arrays are manually added using
	one of the <tt>append(..)</tt> methods, not when using
	reflection.
	**/
	public boolean expandArrays = false;
	
	/** Print the field name ? default: <tt>true</tt>. If 
	field names are <b>not</b> printed, then neither is
	the {@link #FieldAndValSep} - only the value of a field
	is printed.
	**/
	public boolean fieldName = true;

	/** Print the class name at all ? default: <tt>true</tt> **/
	public boolean className = true;
	
	/** print full class name ? default: <tt>false</tt> **/
	public boolean fullClassName = false;
		
	/** print indentity hash code for the object ? default: <tt>true</tt> **/
	public boolean idHashCode	=   true;
    
	/** print field names when using reflection ? default: <tt>true</tt>**/
	public boolean reflectFieldName =	true;

	/** 
	Prints the superclass's variables when using reflection ? default: <tt>false</tt>
	**/
	public boolean reflectSuperClass = false;

	/** 
	Reflects static variables. By default this is <tt>false</tt> 
	since statics are not part of the object's instance-state.
	**/
	public boolean reflectStatics = false;

	/** 
	Default access level when using reflection (fields with
	this or looser access will be printed). default: PRIVATE (EVERYTHING
	IS PRINTED)
	**/
	public VisibleLevel reflectVisibleLevel = VisibleLevel.PRIVATE;
	
	public static final class VisibleLevel 
		{
		private VisibleLevel() { }
		public static VisibleLevel PUBLIC = new VisibleLevel ();
		public static VisibleLevel PROTECTED = new VisibleLevel ();
		public static VisibleLevel DEFAULT = new VisibleLevel ();
		public static VisibleLevel PRIVATE = new VisibleLevel ();
		}    	
	
	public Style() { }
	public String toString() {
		return new ToString(this).reflect().render();  
		}
    
	} //~class Style


//==appends===========================================

/** 
Appends an arbitrary string to the result. This can be
used as a prologue, epilogue etc. 
**/
public ToString append(Object str) 
	{
	result.append(str);
	return this;
	}
	
public ToString append(String fieldName, Object val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);

	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		} 
	result.append(val);

	firstFieldDone = true;
	return this;
    }

public ToString append(String fieldName, String val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);

	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		} 
	result.append(val);

	firstFieldDone = true;
	return this;
    }

//Primitive Types
public ToString append(String fieldName, long val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	result.append(val);

	firstFieldDone = true;
	return this;
    }

public ToString append(String fieldName, int val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	result.append(val);

	firstFieldDone = true;
	return this;
    }

public ToString append(String fieldName, short val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	result.append(val);

	firstFieldDone = true;
	return this;
    }

public ToString append(String fieldName, byte val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	result.append(val);

	firstFieldDone = true;
	return this;
    }

public ToString append(String fieldName, double val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	result.append(val);

	firstFieldDone = true;
	return this;
    }

public ToString append(String fieldName, float val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	result.append(val);

	firstFieldDone = true;
	return this;
    }

public ToString append(String fieldName, char val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	result.append(val);

	firstFieldDone = true;
	return this;
    }

public ToString append(String fieldName, boolean val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	result.append(val);
	
	firstFieldDone = true;
	return this;
    }

//Array types
public ToString append(String fieldName, Object[] val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
		
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	
	if (style.expandArrays) 
		{
		result.append(style.startArray);
		for (int n = 0; n < val.length; n++) 
			{
			if (n != 0) 
				result.append(style.arrayValSep);
			result.append(val[n]);
			}
		result.append(style.endArray);
		}
	else
		result.append(val);

	firstFieldDone = true;
	return this;
    }

public ToString append(String fieldName, long[] val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
		
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	
	if (style.expandArrays) 
		{
		result.append(style.startArray);
		for (int n = 0; n < val.length; n++) 
			{
			if (n != 0) 
				result.append(style.arrayValSep);
			result.append(val[n]);
			}
		result.append(style.endArray);
		}
	else
		result.append(val);

	firstFieldDone = true;
	return this;
    }

public ToString append(String fieldName, int[] val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
		
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	
	if (style.expandArrays) 
		{
		result.append(style.startArray);
		for (int n = 0; n < val.length; n++) 
			{
			if (n != 0) 
				result.append(style.arrayValSep);
			result.append(val[n]);
			}
		result.append(style.endArray);
		}
	else
		result.append(val);

	firstFieldDone = true;
	return this;
    }

public ToString append(String fieldName, short[] val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
		
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	
	if (style.expandArrays) 
		{
		result.append(style.startArray);
		for (int n = 0; n < val.length; n++) 
			{
			if (n != 0) 
				result.append(style.arrayValSep);
			result.append(val[n]);
			}
		result.append(style.endArray);
		}
	else
		result.append(val);

	firstFieldDone = true;
	return this;
    }

public ToString append(String fieldName, byte[] val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
		
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	
	if (style.expandArrays) 
		{
		result.append(style.startArray);
		for (int n = 0; n < val.length; n++) 
			{
			if (n != 0) 
				result.append(style.arrayValSep);
			result.append(val[n]);
			}
		result.append(style.endArray);
		}
	else
		result.append(val);

	firstFieldDone = true;
	return this;
    }

public ToString append(String fieldName, char[] val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
		
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	
	if (style.expandArrays) 
		{
		result.append(style.startArray);
		for (int n = 0; n < val.length; n++) 
			{
			if (n != 0) 
				result.append(style.arrayValSep);
			result.append(val[n]);
			}
		result.append(style.endArray);
		}
	else
		result.append(val);

	firstFieldDone = true;
	return this;
    }
	
public ToString append(String fieldName, double[] val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
		
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	
	if (style.expandArrays) 
		{
		result.append(style.startArray);
		for (int n = 0; n < val.length; n++) 
			{
			if (n != 0) 
				result.append(style.arrayValSep);
			result.append(val[n]);
			}
		result.append(style.endArray);
		}
	else
		result.append(val);

	firstFieldDone = true;
	return this;
    }
	
public ToString append(String fieldName, float[] val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
		
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	
	if (style.expandArrays) 
		{
		result.append(style.startArray);
		for (int n = 0; n < val.length; n++) 
			{
			if (n != 0) 
				result.append(style.arrayValSep);
			result.append(val[n]);
			}
		result.append(style.endArray);
		}
	else
		result.append(val);

	firstFieldDone = true;
	return this;
    }

public ToString append(String fieldName, boolean[] val) 
	{
	if (firstFieldDone)
		result.append(style.fieldSep);
		
	if (style.fieldName) { 
		result.append(fieldName);
		result.append(style.fieldAndValSep); 
		}
	
	if (style.expandArrays) 
		{
		result.append(style.startArray);
		for (int n = 0; n < val.length; n++) 
			{
			if (n != 0) 
				result.append(style.arrayValSep);
			result.append(val[n]);
			}
		result.append(style.endArray);
		}
	else
		result.append(val);

	firstFieldDone = true;
	return this;
    }

//==end appends============================================	

}          //~class ToString