// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util;

import java.util.*;
import java.util.regex.*;
import java.security.*;

/* 
UUID and token utilities

@author hursh jain
*/
public final class UUIDUtil
{
static MessageDigest digester = null;

static {
	try {
		digester = MessageDigest.getInstance("SHA-1");
		}
	catch (Exception e) {
		e.printStackTrace();
		}
	}

/**
Returns a ID suitable for a session/cookie identifier.
<pre>
See: cookies.lcs.mit.edu
See: www.across.si

There are 2 issues with generating sessionid's. 

1) uniqueness - 2 or more sessionid's should not end up being 
   the same.
2) hard-to-guess - For example, sequential values like 
   1, 2, 3 are unique but easy to guess and therefore easy
   to session hijack.

Our sessionid's have 2 parts:
a) a timestamp for guaranteed uniqueness  (easy to guess)
b) random data (hard to guess)
</pre>
*/
public static String newSessionID() 
	{
	//although this function has been written for some time,
	//i recently looked this over with webscarab (around aug 2007)
	//and it visually appears fairly scattered/random. (9999 sid's)
	//dunno, maybe hotbits retrieved/cached every minute via a background
	//thread is good if one wants to be hardcore but it's pretty good
	//as-is 
	
	//there is also no visual difference between nanos & millis. intuitively
	//nanos should have more differences in the lower bits but after
	//all the bits get churned/digested, there is no diff that i can tell,
	//at least visually via webscarab. so i've left this as millis.
	
	final long time = System.currentTimeMillis();   

	Random random = new java.util.Random();
	final int rand = random.nextInt(); 
	final String cookie = String.valueOf(time) + rand;

	byte[] digestbuf = null;
	
	//digests always returns 16 bytes	   			
	synchronized (UUIDUtil.class) { 	
		digestbuf = digester.digest(cookie.getBytes());
    	}
  	
	/*
	we simply convert bytes toHex. note, we don't use Base64
	because '/' and '+' are valid Base64 chars and they can
	mess up url's. 
	*/
    
    char[] encoded = null;
    encoded = encode(digestbuf);
    return new String(encoded);
    
    //note, we don't ever need to use the decode() method
    //but it's there for testing foo->encode->decode->foo
    }


//well, i was very...young...when I wrote this... :-]
//
//all right, as of 2014, i took out the original string, a bit too risque
//new lookup string is less embarassing
//					   		1..............16 [or 0-15, 16 things]	
//					   		0123456789ABCDEF
static final char[] lookup="wInteEMubY2169x0".toCharArray();

/*
base16 encoding
	   0 -> some ascii code
       1 -> some ascii code
       ...
       15-> some ascii code
       
when ascii codes are for 0...F it's called "hex" encoding. 
*/

//we need a reverse lookup table since are ain't
//using straight hexadecimal to encode
//encode: half byte [0-16] -> char value from lookup string
//decode: char value from lookup string (0-128) -> half byte
static final int[] reverselookup = new int[128];
	
static	
	{
	for (int n = 0; n < 16; n++) {
		int x =  lookup[n] & 0xFF; //x ='c'('c'&0xff->99), 'n'(110) etc
		reverselookup[x] = n;  //this is why reverselookup[128] above, not 16
		}
	}
	
private static final char[] encode(byte[] buf) 
	{
	final int len = buf.length;

 	char[] tmp = new char[len * 2];  
	
	for (int n = 0, i = 0; n < len ; n++, i+=2)
		{
		int high4 =  (buf[n] >> 4) & 0x0F;
		int low4  =  buf[n] & 0x0F;
		
		tmp[i]   = lookup [high4];
		tmp[i+1] = lookup [low4];
		}
	return tmp;
	}

private static final byte[] decode(char[] buf) 
	{
	final int len = buf.length;

	byte[] tmp = new byte[(len/2) + (len%2)];

	for (int n = 0, tmp_n = 0; n < len; n+=2, tmp_n++) 
		{
		//0xFF makes it an int. we could cast to (int)
		//too but then that would sign extend the char
		//in the general case [but since all are chars
		//are < 127], it would be ok to do so in this
		//case. Used 0xFF because that's the right thing
		//generally speaking

		int c1 = buf[n] & 0xFF;   //'w', 'I', 'n', 't' etc.. 
		int c2 = buf[n+1] & 0xFF;

		if ( c1 > 127 || c2 > 127)
			throw new IllegalArgumentException("malformed bufing, encoded with characters I don't understand [" + c1 + "] and [" + c2 + "]");
				
		int i = (reverselookup[c1] << 4  | reverselookup [c2]); 
		//System.out.println( (reverselookup[c1] << 4) + ", " + reverselookup[c2] + ",->" + i);

		tmp[tmp_n] = (byte) i;
		}
	return tmp;
	}
	

static char[] URL_safe_chars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ123456789_-.~".toCharArray();

/**
Returns a N-char string suitable for tracking, affiliate, short codes, etc. Uses
characters safe for all URL's.

@param	len	the length of the returned string
*/
public static String newID(int len)
	{
	final Random r = new Random();
	final char[] buf = new char[len];
	for (int n = 0; n < len; n++) {
		buf[n] = URL_safe_chars[r.nextInt(URL_safe_chars.length)];
		}
	
	return new String(buf);
	}

/**
Returns a 8 character string suitable for tracking, affiliate, short codes, etc. Uses
characters safe for all URL's.
*/
public static String newID()
	{
	return newID(8);
	}
		
public static void main (String args[])
	{
	Args myargs = new Args(args);
	int LOOP = myargs.getInt("loop", 100000);
	myargs.setUsage("java fc.util.UUIDUtil [-loop <int, default 100,000>]");
	
	if (myargs.flagExists("help")) {
		myargs.showError();
		return;
		}
	System.out.println("Session ID: " + newSessionID());
	System.out.println("Generating " + LOOP + " test session ids...");

	fc.util.Watch w = new fc.util.Watch();
	Map map = null;
	int count = -1;
	
	w.start();	
	for (int n = 0; n < LOOP; n++) {
		String s = newSessionID();
		}
	System.out.println("Time for " + LOOP + ": " + w.time() + " ms");
	
	System.out.println("Now testing for collisions...");
	w.restart();

	map = new HashMap();
	count = 0;
	for (int n = 0; n < LOOP; n++) 
		{
		String s = newSessionID();
		if (map.containsKey(s)) {
			System.out.println("collision found with key: " + s);
			count++;
			}
		if (n > 0 && n % 20000 == 0) {
			w.stop();
			System.out.println("generated: " + n + " (+ " + w.time() + " ms)");
			w.restart();
			}
		map.put(s, null);
		}
	if (count == 0) {
		System.out.println("No collisions found...");
		}
	else{
		System.out.println(count + " collisions found...");
		}

	System.out.println("------------------------------");
	w.restart();	
	System.out.println("newID (8): " + newID(8));
	System.out.println("Generating " + LOOP + " test session ids...");

	for (int n = 0; n < LOOP; n++) {
		String s = newID(8);
		}
	System.out.println("Time for " + LOOP + ": " + w.time() + " ms");

	System.out.println("Now testing for collisions...");
	w.restart();

	map = new HashMap();
	count = 0;
	for (int n = 0; n < LOOP; n++) 
		{
		String s = newID(8);
		if (map.containsKey(s)) {
			System.out.println("collision found with key: " + s);
			count++;
			}
		if (n > 0 && n % 20000 == 0) {
			w.stop();
			System.out.println("generated: " + n + " (+ " + w.time() + " ms)");
			w.restart();
			}
		map.put(s, null);
		}
	if (count == 0) {
		System.out.println("No collisions found...");
		}
	else{
		System.out.println(count + " collisions found...");
		}
	}
	

}