// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util.cache;	

import java.text.*;
import java.util.*;
import fc.io.*;
import fc.util.*;

/**
Memory resident cache of objects. Allows to set expire times for contained
objects after which those expired objects are removed. By default, references
to expired objects are not retained in the cache, such that those object may
be garbage collected as needed by the garbage collector (if those objects
becomes unreachable from the rest of the program).
<p>
The size of this cache is unbounded. To use a bounded cache, use 
the {@link BoundedCache} class. 
<p>
Expired items are reaped via a background high priority thread that sleeps
for intervals between expired item checks. After every reap operation, the
reaper thread sleeps for 120 seconds (by default). This implies that the
minimum expire time for any item will be 120 seconds. This sleep time can
be changed via the {@link #setReapInterval} method.
<p>
ThreadSafety: This class <b>is</b> thread safe and can be used by multiple 
threads concurrently.

@author 	hursh jain
@version 	1.0 12/29/2001
**/
public final class MemoryCache implements Cache
{

/** Useful constant of 1 second (in milliseconds) */
public static final long ONE_SEC = 1000;

/** Useful constant of 1 second (in milliseconds) */
public static final long ONE_SECOND = ONE_SEC;

/** Useful constant of five seconds (in milliseconds) */
public static final long FIVE_SEC = 5 * ONE_SEC;

/** Useful constant of five seconds (in milliseconds) */
public static final long FIVE_SECOND  = FIVE_SEC;

/** Useful constant of ten seconds (in milliseconds) */
public static final long TEN_SEC  = 10 * ONE_SEC;

/** Useful constant of ten seconds (in milliseconds) */
public static final long TEN_SECOND = TEN_SEC;

/** Useful constant of 30 seconds (in milliseconds) */
public static final long THIRTY_SEC  = 30 * ONE_SEC;

/** Useful constant of 30 seconds (in milliseconds) */
public static final long THIRTY_SECOND 	= THIRTY_SEC;

/** Useful constant for cache expiry time of 1 minute (in milliseconds) */
public static final long ONE_MIN  = 1  * 60  * 1000;

/** Useful constant for cache expiry time of 2 minutes (in milliseconds) */
public static final long TWO_MIN = 2  * 60  * 1000;

/** Useful constant for cache expiry time of five minutes (in milliseconds) */
public static final long FIVE_MIN = 5  * 60  * 1000;

/** Useful constant for cache expiry time of ten minutes (in milliseconds) */
public static final long TEN_MIN = 10 * 60 * 1000;

/** Useful constant for cache expiry time of thirty minutes (in milliseconds) */
public static final long THIRTY_MIN = 30 * 60 * 1000;

/** Useful constant for cache expiry time of one hour (in milliseconds) */
public static final long ONE_HOUR = 60 * ONE_MIN;

/** Useful constant for cache expiry time of two hours (in milliseconds) */
public static final long TWO_HOUR = 2  * ONE_HOUR;

/** Useful constant for cache expiry time of four hours (in milliseconds) */
public static final long FOUR_HOUR 	= 4  * ONE_HOUR;

/** Useful constant for cache expiry time of eight hours (in milliseconds) */
public static final long EIGHT_HOUR = 8  * ONE_HOUR;

final		String 				myName;
final		Map 				cache;
volatile 	boolean 			closed = false;
volatile 	long 				defaultTTL;
volatile 	long 				reapInterval;
			CacheReaperThread 	reaperThread;
final		Log					log;

/**
Instantiates this class with the specified name and logger.

@param  name		String denoting the name for this object
@param	logger		a reference to a {@link fc.io.Log}
@param	default_ttl	the default time to live for objects in the cache (in milliseconds)
					(objects may appear to live for greater than the ttl if the the reaper
					interval time is greater than this value). Also note, that this is the
					<i>default</i> time and that the {@link #put(Object,Object,long) put}
					method allows a different ttl for each individual object.
**/
public MemoryCache(Log log, String name, long default_ttl_millis) 
	{
	if (log == null)
		log = Log.getDefault();
	
	this.log = log;
	this.myName = name;
	this.defaultTTL = default_ttl_millis;
	this.reapInterval = 120 * 1000;
	this.cache = Collections.synchronizedMap(new HashMap());
	this.reaperThread = new CacheReaperThread(); 
	}

/**
Creates a memory cache with a system-assigned logger and name and
a default ttl of 30 minutes.
*/
public MemoryCache() 
	{
	this(null, 
		"MemoryCache/created@" 
		+ DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.SHORT)
			.format(new Date()), 
		30 * 60);
	}

/** 
Sets the minimum time between expired item checks.
**/
public void setReapInterval(final long interval) {
	reapInterval = interval;
	}

/** 
Sets the cache reaper thread priority. By default the reaper runs at MAX_PRIORITY - 1 but
if there are a lot of low priority caches, setting this to a lower priority could be useful.
**/
public void setReapThreadPriority(final int priority) {
	reaperThread.setPriority(priority);
	}


public boolean containsKey(Object key) 
	{
	Argcheck.isfalse(isClosed(), "Memory cache has been closed");

	final CacheItem item = (CacheItem) cache.get(key);	

	if (item == null) {
		return false;
		}
		
	 if (item.hasExpired()) {
		return false;
		}

	return true;
	}


public Object get(Object key) 
	{
	Argcheck.isfalse(isClosed(), "Memory cache has been closed");

	final CacheItem item = (CacheItem) cache.get(key);	

	if (item == null) 
		return null;
		
	 if (item.hasExpired()) {
		return null;
		}
	 else {
		return item.getValue();
		}
	}

public Map getAll() {
	return cache;
	}

public long getTimeLeft(Object key) 
	{
	final CacheItem item = (CacheItem) cache.get(key);	

	if (item == null) 
		return 0;
	
	if (item.expire == -1) 
		return -1;
	
	final long time = item.calcTimeLeft();
	
	if (time <= 0)
		return 0;
		
	return time;
	}

public Object put(Object key, Object val, long expiry)
	{
	Argcheck.isfalse(isClosed(), "Memory cache has been closed");
	Argcheck.istrue(expiry >= 0 || expiry == -1, "Illegal value [" + expiry + "] for the expiry argument, need the value to be >=0 or -1");
	
	
	final CacheItem cacheitem = new CacheItem(key, val, expiry);
	final Object item = cache.put(key, cacheitem);			
	if (item != null)  {  //return previous item
		return ((CacheItem)item).getValue();
		} 
	return null;
	}

public Object put(Object key, Object item)
	{
	Argcheck.isfalse(isClosed(), "Memory cache has been closed");
	return put(key, item, getDefaultTTL());
	}

public long expireTimeLeft(Object key)
	{
	Argcheck.isfalse(isClosed(), "Memory cache has been closed");
	final Object item = cache.get(key);	
	if (item == null) {
		return 0;
		}
	else {
 		return ((CacheItem)item).expireTimeLeft();
		}		
	}

public void expire(Object key)
	{
	Argcheck.isfalse(isClosed(), "Memory cache has been closed");
	final Object item = cache.get(key);	
	if (item != null) {
		final CacheItem citem = (CacheItem)item; 
		if ( ! citem.hasExpired() ) {  //test not needed but conceptually clean
			citem.expireNow();  //expire now
			}
		}
	}

public void extend(Object key, long renewTime)
	{
	Argcheck.isfalse(isClosed(), "Memory cache has been closed");
	Argcheck.istrue(renewTime >= 0 || renewTime == -1, "Illegal value [" + renewTime + "] for the renewTime argument, need the value to be >=0 or -1");
	final Object item = cache.get(key);	
	if (item != null) {
		final CacheItem citem = (CacheItem)item; 
		if (citem != null) {
			if ( ! citem.hasExpired() ) {
				citem.renew(renewTime);
				}
			}
		}
	}

/**
Closes this cache, which makes all items in the cache unavailable. Any items
needed for later should be taken out of the cache before closing it.
<p>
<b>Note:</b>, it is a good idea to close the cache once it's not needed,
because it releases resources, including any internal threads spawned and used
by this implementation.
**/
public void close()
	{
	reaperThread.stopThread();
	this.closed = true;
	log.info("*** cache:[", myName, "] closed. ***"); 
	}

/**
Returns true if this cache has been closed, false otherwise.
**/
public boolean isClosed()
	{
	return this.closed;
	}

/**
Sets the default TTL in milliseconds.
*/
public void setDefaultTTL(long millis) {
	defaultTTL = millis;
	}

/**
Gets the default TTL in milliseconds.
*/
public long getDefaultTTL() {
	return defaultTTL;
	}

public void clear() {
	//don't clear in the middle of reaper thread or getAll iteration
	synchronized (cache) { 
		cache.clear();
		}
	}

public String toString()
	{
	final int size = cache.size();
	String temp = this.myName + "; contains ";
	temp += (size == Integer.MAX_VALUE) ? 
			Integer.toString(size) 	: 	Integer.toString(size) + " (or greater)";
	temp += " items; ";
	temp += isClosed() ? "cache is closed. " : "cache is open.";
	return temp;
	}

//stores key,val pairs along with expiry data, never expires if expiry is -1
private class CacheItem
{
final Object 	key;
final Object 	val;
final long 		startTime;
	  long 		expire;
	  long 		lastAccessTime;

CacheItem(final Object key, final Object val, long expire)	
	{
	this.key = key;
	this.val = val;
	this.expire = expire;
	this.startTime = System.currentTimeMillis();
	this.lastAccessTime = 0;
	}

boolean hasExpired() 
	{
	if (this.expire == -1) {
		return false;
		}
	else {
		return (calcTimeLeft() <= 0);
		}
	}

void renew(long time)
	{
	if (this.expire == -1)	{
		this.expire = 0; //so that we don't lose 1 millisec between -1 and 0
		}
	this.expire = this.expire + time; //this also works if time == -1
	}
	
void expireNow()
	{
	this.expire = 0;
	}
	
long expireTimeLeft()
	{
	if (this.expire == -1) {
		return 0;
		}
	else return calcTimeLeft();
	}

long calcTimeLeft()
	{
	return (this.expire + this.startTime) - System.currentTimeMillis();
	}

Object getValue()
	{
	this.lastAccessTime = System.currentTimeMillis();
	return this.val;
	}

/**
Returns a description but the exact details of said description
are unspecified and subject to change. However, the following
may be regarded as typical:
<tt>name; key=value; expired=true/false</tt>
**/
public String toString()
	{
	return "(MemoryCache.CacheItem: Expired=" + hasExpired() + ";<" + this.key + ">=<" + this.val + ">)";
	}

}	//~inner class CacheItem

//instantiate/start only 1 reaper thread per cache. This could also have
//been achieved using java.util.Timer/TimerTask
final class CacheReaperThread extends Thread
{
volatile boolean stop = false;

public CacheReaperThread()
	{
	setName("CacheReaperThread->["+myName+"]");	
	setPriority(Thread.MAX_PRIORITY - 1); //does it's work asap while it's awake.
	setDaemon(true);
	start();	
	}
	
public void run()
	{
	log.bug("CacheReaperThread for cache:[", myName, "] started..."); 
	while (! stop)
		{
		try {
			synchronized(this) 
				{ 
//				log.bug("reaperthread: begin sleep for: ", reapInterval, " milliseconds"); 
				this.wait(reapInterval); 
				}
			}
		catch (InterruptedException e) {
			//e.printStackTrace();
			break;
			}
		log.bug("CacheReaperThread for cache:[", myName, "] now running reapCache()"); 
		reapCache();			
		}
	}

public void stopThread()
	{
	this.stop = true;
	//might be waiting for the reapInterval duration so also interrupt the thread
	this.interrupt();
	}

final void reapCache()
	{
	log.bug("waiting for memorycache object lock"); 
	synchronized (cache) 
		{
		log.bug("acquired lock for cache, now running reap loop"); 
		final Iterator it = cache.entrySet().iterator();
		while (it.hasNext()) { 
			CacheItem citem = (CacheItem)(((Map.Entry) it.next()).getValue());
			if (citem.hasExpired()) {
				log.bug("Reaping: ", citem); 
				it.remove();	
				}
			}
		}	
	}

}	//~inner class CacheReaperThread

public static void main(String[] args)
	{
	new Test(new Args(args));
	}

static private class Test
	{
	Test(Args myargs) 
		{
		try {
			Cache mycache = new MemoryCache();
			((MemoryCache)mycache).setReapInterval(5 * 1000);

			((MemoryCache)mycache).log.setLevel(myargs.get("log", "debug"));
			
			mycache.put("key1", "val1");
			mycache.put(null, "val2");
			mycache.put("key3", null);
			mycache.put(null, null);
			System.out.println("key1 = " + mycache.get("key1"));
			System.out.println("key2 = " + mycache.get("key2"));
			System.out.println("key3 = " + mycache.get("key3"));
			int num = 10;
			System.out.println("adding: " + num + " key,val pairs");
			for (int n = 0; n < num; n++) {
				mycache.put("key" + n, "val" + n); 
				}
			System.out.println("finished adding. cache now contains the following"); 
			for (int n = 0; n < num; n++) {
				System.out.println("key"+n + "=" + mycache.get("key" + n)); 
				}
			System.out.println("Now expiring every other item");
			for (int n = 0; n < num; n++) {
				if (n % 2 == 0) 
					continue;
				mycache.expire("key"+n);
				}

			System.out.println("Expire times for cache entries:"); 
			for (int n = 0; n < num; n++) {
				System.out.println("key"+n + "=" + mycache.getTimeLeft(("key"+n))); 
				}

			System.out.println("Sleeping for 30 seconds...give reaper time to do it's thing.");
			Thread.currentThread().sleep(30*1000); //give reaper thread time to work
			
			System.out.println("Expiring finished, cache now contains the following (expired keys should return null)");
			for (int n = 0; n < num; n++) {
				System.out.println("key"+n + "=" + mycache.get("key" + n)); 
				}
				
			System.out.println("Expire time for 'key1' is: " + mycache.getTimeLeft("key1"));
			System.out.println("Expire time for 'key2' is: " + mycache.getTimeLeft("key2"));
	
			System.out.println("closing cache");
			mycache.close();
			System.out.println("mycache.toString() = " + mycache);
			System.out.println("the following should throw an Exception");
			mycache.put("foo", "bar"); 	
			}
		catch (Exception e) {
			e.printStackTrace();
			}
		} //~init
	} //~class test

}			//~MemoryCache
