// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util.pagetemplate;

import java.util.*;
import java.io.*;

import fc.io.*;
import fc.util.*;

/**
A classloader that loads templates. Each page is loaded by it's own 
separate ClassLoader.
<p>
If a page changes on disk (this is tracked by {@link TemplateMgr}), then the
new page is loaded by a new instance of this class.
@author hursh jain
*/
public class TemplateClassLoader extends ClassLoader 
{
/*
We don't need to use definePackage for our classloaders,
there are no manifest based versions associated with pages.

Say a java file in package foo.bar exists in: 
	/mydir/foo/bar/baz.java

Then running "javac -classpath /mydir foo.bar.baz.java" will 
result in: 
	/mydir/foo/bar/baz.class

Now say: 
	cd /mydir/foo/bar/
	java -classpath "." foo.bar.baz
	
This will cause an error since "java" actually checks to
see that the classpath it _used_ is the same as the classpath
in the file it found. (in this example, java used "." but found
foo.bar.baz.class).

However, we can read a byte[] corresponding to a .class file
from anywhere on the disk, network, database and this has 
(typically for custom loaders) nothing to do with the classpath.

Therefore, when implenting a custom classloader, we can
maintain a directory structure for generated class files OR
put all generated class files in one directory, as say
/mydir/foo_bar_baz.class etc OR do both.

GENERATED TEMPLATE CLASSES ARE ALL IN THE SAME PACKAGE (molly.pagetemplate) AND
ARE ORGANIZED IN A DIRECTORY STRUCTURE THAT MIRRORS THE SOURCE
HEIRARCHY. THIS MAKES GENERATED .java PAGES (CORRESPONDING TO THE SOURCE
.mp) EASIER TO FIND FOR HUMANS.

Instead of simply implementing findClass, we _could_ also 
override loadClass(). This allows us to warn/error out if
the parent classloader was able to load the page's class
file (which should never happen since the classpath containing
the template .class files should NOT be in the system classpath).
If it IS in the system classpath, page re-loading will not
work, which is an error for our purposes. 

We still need to delegate to a parent classloader for non-template
classes referenced from a page. To simplify things, I've used
the delegation model here and will give the parent classloader
first crack at loading everything....this is the way it should
always be anyway.
*/

private static final boolean dbg = false;
Log		log;

/**
@param	scratch 		absolute path to the scratch dir
@param	log				logging destination.
*/
public TemplateClassLoader(Log log) 
	{
	this.log = log;
	}

/**
Calls {@link TemplateClassLoader} with the logger set to:
<tt>fc.web.page.TemplateClassLoader</tt>
*/
public TemplateClassLoader() 
	{
	this.log = Log.getDefault();
	}

Map class_file_paths = new HashMap();

/**
Loads a class corresponding to a page (i.e., a name that starts
with {@link Page.PackageName} from. Delegates the loading of all 
other classes to the parent classloader (typically the system
classloader).
<p>
If the page suffix is <tt>.mp</tt>, then a name such as
<tt>foo/bar/baz.page</tt> is loaded from the scratch dir (from
<tt><scratchdir>/foo/bar/baz.class</tt>).

@param	name	a relative page path-name (relative to the page 
				root directory), for example <tt>foo/bar/my.page</tt>
				or <tt>./my.page</tt>
*/
public Class loadClass(String name, boolean resolve) throws ClassNotFoundException
	{
	if (dbg) log.bug("enter loadClass(): name=", name, " resolve=", resolve);
	
	//we could implement findClass() instead of loadClass
	if (name.startsWith(TemplatePage.PACKAGE_NAME))
		{
		Class c = findLoadedClass(name);
		if (c != null)
			return c ;
			
		if (dbg) log.bug("Name starts with:[", TemplatePage.PACKAGE_NAME, "], loading page: ", name);
		final int pkgnamelen =  TemplatePage.PACKAGE_NAME.length() + 1; //"foo.pkg."
	
		name = name.substring(pkgnamelen, name.length());
	
		if (dbg) System.out.println("[A] name=" + name);
		//for inner, anonymous classes
		//1) pagemgr first says: load  "molly.page.foo.class"  
		//2) foo.mp contains a anonymous inner class foo.mp$1
		//3) this method is called again with "molly.page.foo.mp$1" 
		//	 as the argument
		//4) we need to load "foo.mp$1.class"
	
		if (! name.endsWith(".class")) 
			{
			if (dbg) System.out.println("[B]");
			String outer_class_name = name.substring(0, name.indexOf("$"));
			if (dbg) System.out.println("[C] outer=" + outer_class_name);
			String path = (String) class_file_paths.get(outer_class_name);
			if (dbg) System.out.println("[D] path=" + path);
			if (path == null) {
				throw new ClassNotFoundException("Internal error: cannot load anonymous/inner classe: " + name);
				}	
	
			name = path + name + ".class";
			if (dbg) log.warn("anonname=",name);
			}
		else{		
			if (dbg) System.out.println("[E]");
			String dirname = StringUtil.dirName(name); //has ending '/'
			String filename = StringUtil.fileName(name);
			filename = filename.substring(0, filename.indexOf(".class"));
			class_file_paths.put(filename, dirname);
			if (dbg) log.bug("name=",name);
			if (dbg) log.bug("dirname=",dirname);
			}
			
		c  = readClassFromDisk(new File(name));
		
		if (resolve) {
			resolveClass(c);       
			}
		return c;
		}
		
	if (dbg) log.bug("Delegating to thread-context classloader to load class: ", name);
	//we cannot use the parent classloader because it will not have
	//classes inside of WEB-INF (the parent classloader is a jvm level
	//classloader). We need to use the servlet containers classloader
	//instead.
	//return super.loadClass(name, resolve);
	return Thread.currentThread().getContextClassLoader().loadClass(name);
	}

Class readClassFromDisk(File classfile) throws ClassNotFoundException
	{
	Argcheck.istrue(classfile.exists(), "The specified file does not exist: " + classfile.getAbsolutePath());
	
	byte[] buf = null;
	
	try {		
		if (dbg) log.bug("Using filename:", classfile);
		
		buf = IOUtil.fileToByteArray(classfile);
		
		if (buf == null) {
			throw new ClassNotFoundException(classfile.getPath());
			}
			
		} 
	catch (Exception e) {
		throw new ClassNotFoundException(classfile.getPath());
		}
	
	/* 
	definePackage(Page.PACKAGE_NAME, null, null, null, null, null, null, null);

	this works ONCE per instance of the classloader -- will crap out
	if we use the same instance to define more than 1 class. defining
	a package is not needed at all, but may be nice to have for logging
	purposes/setting logging levels. If we have an inner class, this
	method will be invoked more than once (one per inner class which
	uses this same classloader). 
	
	All of this is very obscure and if this causes problems, remove.
	*/
	if (getPackage(TemplatePage.PACKAGE_NAME) == null)
		definePackage(TemplatePage.PACKAGE_NAME, null, null, null, null, null, null, null);

	Class c = defineClass(null, buf, 0, buf.length);
	
	return c;
	}

public static void main (String args[])  throws Exception
	{
	Args myargs = new Args(args);
	myargs.setUsage("java " + ClassUtil.getClassName() + 
		" -file full-path-to-class-file");
	String page = TemplatePage.PACKAGE_NAME + "." + myargs.getRequired("file");
	
	TemplateClassLoader cl  = new TemplateClassLoader();
	cl.log.setLevel("debug");
	
	System.out.println("Loading java.lang.object: ");
	Class c = cl.loadClass("java.lang.Object");
	System.out.println(ClassUtil.getClassLoaderInfo(c));

	System.out.println("Loading " + page);	
	c = cl.loadClass(page);
	System.out.println(ClassUtil.getClassLoaderInfo(c));
	System.out.println("Package for this class: " + c.getPackage());
	}

}
