// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.util.pagetemplate;

import java.io.*;
import java.util.*;
import java.sql.*;

import fc.util.*;
import fc.io.*;

/** 
A superclass for generated template pages. All templates derive from this class. Runs
outside the web environment (intended to be invoked from the command line)
<p>
Pages are always assumed to be written in UTF-8 (a superset of regular
ascii/IS0-8859-1). If the template page is sent as HTML to a device (via some outside
mechanism that uses the templates as part of its workflow), then the HTML contained in
the page template should set the appropriate content type/encoding meta header to say
UTF-8, <b>if</b> UTF-8 characters (such as emojis) are used in the page.
<p>
See {@link TemplateMgr}

@author hursh jain
*/
public abstract class TemplatePage 
{
static private	final boolean internal_dbg = false;

public static String 	PACKAGE_NAME = "molly.pagetemplate";
public static String 	DEFAULT_ENCODING 		= "UTF-8";

/** 
Controls whether debugging is on or off. For expensive debug statements,
<code>if (dbg) bug(....)</code> type statements can be used in a page.
*/
protected volatile 	boolean  dbg = false;

public Log	  	log = Log.getDefault();	
public  Writer 	out;

public File templateFile;

public Connection con;
public Map context;

public void setSourcePath(File templateFile) {
	this.templateFile = templateFile;
	}

public String getSourcePath()
	{
	return templateFile.getAbsolutePath();
	}

/* set a connect that can later be used in render, if needed. code in render can also get
   a connection directly from a connection mgr, etc as well, of course. The resposibility
   to close the connection can be inside render() or by the caller, whatever makes sense
   in a particular use case.
   */
public void setConnection(Connection con) {
	this.con = con;
	}
	
/* set various objects (via a map) that can be later used in render, if needed. */
public void setContext(Map m) {
	this.context = m;
	}
	
public void render(Writer out) throws Exception
	{
	//writers should already have a charset specified if they are using an outputstreamwriter or
	//some other output stream. toPrinteWriter() just wraps that

	render(IOUtil.toPrintWriter(
			IOUtil.bufferWriter(out)));
	}

public void render(File out) throws Exception
	{
	render(new PrintWriter(out, DEFAULT_ENCODING));
	}

public void render(OutputStream out) throws Exception
	{
	render(new PrintWriter(
			new BufferedWriter(
				new OutputStreamWriter(out, DEFAULT_ENCODING))));
	}


/* 
This is the method that runs the templates and writes the generated contents to the
specified output destination. Is implemented by generated classes. 
*/
public abstract void render(PrintWriter out) throws Exception;

/*
Starts/stop debugging with no dbg_prefix/dbg_suffix.

@param	val		true to enable debugging, false to disable.
*/
public final void dbg(final boolean val)
	{
	this.dbg = val;
	}


/**
Prints a debug statement if debugging is turned on for this page.

<p>Typically the implicit page printwriter (the <code>out</code>
variable) will be passed to this method and debug statements will be
printed at the point where they are lexically invoked at the page.
*/
public final void bug(final Writer writer, final Object str1) throws IOException
	{
	if (! dbg)
		return;
	
	writer.append(str1 != null ? str1.toString() : "null");
	}
		
public final void bug(final Writer writer, final Object str1, final Object... args) 
throws IOException
	{
	if (! dbg)
		return;
		
	writer.append(str1 != null ? str1.toString() : "null");
	final int len = args.length;
	for (int i = 0; i < len; i++) {
		writer.append(args[i] != null ? args[i].toString() : "null");
		}
	}

}		
