// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.*;
import java.sql.*;
	
import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/** 
A dependency changes the values or state of some form element based
on the user selection or choice of some <i>other</i> form element. 
<p>
A typical example includes filling later select/popup values based on 
the user chosen value of an earlier select in the form. So, for example,
if the user chooses a country, then the later select shows the states
just for that country. 
<p>
There are several ways to handle dependency changes:
<ol>
<li>The data for some fields (like database based lookup tables) can be
looked up every time. 
<li>All possible options can be looked up ahead of time and then a
subset displayed accordingly. This can be done on the server side (on
form submit) or by client side javascript (using various javascript
arrays containing different sets of data, the arrays are written out
by the server as part of the page and swapped out by client side js).
<p>
Orthogonal to all this is field and form validation. Validation only
makes sense <b>after</b> we have displayed the updated fields to the user
For example, initially a country popup may be empty and a state popup
may be empty. When the user selects a country, then the form is submitted.
At that point, the states popup is still empty and we should *not* validate
the states data (i.e., not show the user a message that a state must be
chosen or show a validation error for any other fields which the user hasn't 
filled out yet). 
<p>
Validation can always be controlled by enabled/disabled flag on the field
since field validators are not run for fields that are disabled. In addition,
any time a field has new dependent values, the form is not validated by field
validators. 

@author hursh jain
**/
public interface Dependency
{

/**
Sets new values for target field in the FormData object. The target field
then uses those values to render itself to the client. The data should be
saved in FormData with the field name as the key.
<p>
<b>Note:</b>This method <blink>must</blink> invoke {@link FormData#setDependencyUpdated}
if it sets new values in the formdata object. New values are values that
are different from the values that were last displayed to the client.
<p>
There are 2 conditions that typically must be satisfied for this to be
true:
<ol>
<li><i>Source not null</i>: The source fields must have some data. For
example, a country popup needs to be selected to some country value before
the target dependent field (a state popup) can be populated. If no country
is selected, we cannot calculate a dependent state value.
</li>
<li><i>Source changed:</i> The source fields must have different data than
the last time the form was rendered (which means that the client made a
different selection/choice). For example, previously, the source field
could have been 'usa'. The client then either changes that to 'uk' or
leaves it alone. The last_value (which is rendered as a hidden field in the
form) in conjunction with the submittd value of the source field(s) can be
used to figure out if there was a dependency change. If the source value is
'usa' and the last_value is 'usa', then we know there is no need for any
dependency updates (since the usa values are already shown to the user).
</li>
</ol>
*/
public void updateValues(FormData fd, HttpServletRequest req) throws DependencyException;

/**
The inverse of {@link renderDependencyData}, sets any depedency data submitted
as part of the form. This data is set in the specified {@link FormData} object
and can be later used by the {@link updateValues} method.
*/
public void setDependencyDataFromSubmit(FormData fd, HttpServletRequest req);

/**
Writes out data to keep track of dependency state. This data is typically
the last selected values of some source field (say last selected source 
country for a target dependent state field) and usually written out as a
hidden field in the form.
*/
public void renderDependencyData(FormData fd, Writer writer) 
throws SQLException, IOException;

/**
Returns the initial values for the target field  These are the values that
are displayed when the form is first rendered. The returned object is of a
type that is suitable for the data of the target field. (typically an instance
of a static inner data class of that field).
<p>
If there is no initial data, the returned object can also be <tt>null</tt> or
an empty data object, depending on what the target field expects when there
is no data.
*/
public Object getInitialValues(Field f);

/**
Returns true if the submitted values are in a valid range of values. 
For ultra-secure applications, the validity can be computed every time by
recomputing/retrieving the dependent data and of-course, database
constraints should always be employed.
<p>
If this method is not applicable, this method should return
<tt>true</tt> every time.
*/
public boolean	isSubmitDataValid(FormData fd);
}

