// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.*;
import java.sql.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/** 
Represents an HTML form field. Subclasses provide concrete representations
of various HTML input types.
<p>
Note: A {@link FieldValidator} for a given field is added to that field
automatically by a new Validator when that validator is instantiated (so a
public <tt>addValidator</tt> method is not needed in this class).
<p>
Fields are normally created and added to a form once. However, new fields can
also be created per user/per-submit and added to the {@link FormData} object.
These fields can then be rendered via the {@link Form#renderDynamicFields}
method and are shown <i>just</i> for that request. They are not part of the form and
cannot be later retrieved by {@link Form#get} -- use {Form#getSubmittedField} 
instead.
<p>
Initial values (if any) for fields are typically specified during form
instantiation. These default values are initially shown to all users when
the form is rendered. However, in some cases, it is possible to show
different initial values to different users. This is done by creating a new
{@link FormData} object before the form is rendered and calling the
<tt>setValue(FormData, ...)</tt> methods on various fields to set values
on the formdata object. 
<p>
FormData objects are automatically created to store submitted form data
(and maintain form state) by the {@link Form#handleSubmit} method. By manually
creating a new FormData object before showing the form, and the using that
object to render the form, per-user initial values can be displayed (this
simulates a user form submission which of course is different for each
user).
<p>
This form api is complex. For a simpler bare-bones alternative, see
{@link fc.web.simpleforms}

@author hursh jain
**/
public abstract class Field
{
static protected SystemLog  log;

protected Form				form;
protected String 			name;
protected List	 			arbitraryString;
protected boolean 			renderStyleTag;
protected String			styleTag;
protected String			label;
protected List 				validators;
protected FieldRefresher	refresher;
protected boolean			enabled = true;

/** 
Creates a new form Field. By default, a field is enabled.
To disable a field (so that it will not be validated),
use the {@link #setEnabled} method.

@param	fieldName	 		the name of this field
**/
public Field(String fieldName)
	{
	Argcheck.notnull(fieldName, "fieldName parameter was null");
	this.name = fieldName;
	log = Log.getDefault();
	renderStyleTag = true;
	styleTag = this.name;
	validators = new ArrayList();
	arbitraryString = new ArrayList(8);
	}
		
/**  
Not used any more, replaced by getTypeXXX methods

Returns the values that were last selected or submitted. The returned
object's runtime type will depend upon the subclass and could be, for
example, a String, String[] or a List. (the subclass should document the
returned type) abstract public Object getValue();
**/

/** Returns the name of this form element **/
public String getName() {
	return name;
	}		
 
/** 
Subclasses should return an appropriate {@link Field.Type}. This type
is rendered as part of <tt>&lt;input type= ...</tt>
**/
abstract public Field.Type getType();	

/**  
This method sets the value of this field from the parameters obtained from
the specified request. The name of the parameter to obtain this value will
typically be the name of this field itself.

@param	fd		the form data object to store the value in
**/
abstract public void setValueFromSubmit(FormData fd, HttpServletRequest req)
throws SubmitHackedException;
 
/**
Returns <tt>true</tt> if this field was isFilled out or selected by the
user, <tt>false</tt> otherwise.
<p>
Note: Some fields like selects will never be empty since non-multiple
select fields always send their default selected value. [although
select/with/multiple can be empty since the browser sends (much like radio
buttoms) nothing at all when no option is selected].
*/
abstract public boolean isFilled(FormData fd);


public abstract void renderImpl(FormData fd, Writer writer) 
throws SQLException, IOException;

/** 
This method writes the HTML for this form field to the given writer. Only
the HTML for the field itself should be written and the enclosing HTML for
this field is left upto the calling code. (typically a dynamic server side
page). The field should be rendered in a sticky fashion, i.e., should
incorporate/display the last submitted value.
<p>
<b>Important Note:</b> Although each field renders itself, validation
errors are not rendered as part of this rendering. This is specifically
designed for flexibility in HTML layout and display of error messages. This
does imply that page authors <font color=blue><b>must</b></font> remember
to extract and write error messages from each field.
<p>
<b>Important Note 2:</b> If providing a non-null argument for the FormData
parameter, this method should only be called after calling {@link
Form#handleSubmit}. FormData objects hold the results of submitted form
data. <b>If the formdata is <tt>null</tt>, it implies that the form is
shown to the user for the first time. The fields then render themselves
based on their default/initial values.</b>. However, one can also display
different initial values unique to a user (say by retrieving their last
filled values from a database). In that case, a FormData object should be
initially/manually created and passed to this method.

@param	fd		the current form data object (can be null). 
@param	writer 	the output destination
**/
public void render(final FormData fd, final Writer writer) throws SQLException, IOException
	{
	if (refresher != null) 
		{
		/* always refreshers are request specific and hence need a valid fd */
		if (fd == null)
			log.info("Looks like Field:[", name, "] is being rendered for the first time. Attached ", refresher, " will not be invoked. Remember to construct/init the field with an appropriate value to see initial values.");
		else 
			refresher.refresh(fd);
		}
		
	renderImpl(fd, writer);
	}

/**
Convenience method that calls <tt>render(null, writer)</tt>
(for when the form needs to be displayed the first time 
without any formdata)
*/
public void render(Writer writer) throws SQLException, IOException {
	render(null, writer);
	}

/**
Similar to {@link render(Writer)} but also renders the specified
prefix/suffix strings before and after each element. For example, one can
say (in a jsp):
<blockquote>
<code><td><% form.get("foo").render(); %><br></td></code>
</blockquote>
<i>or</i>:
<blockquote>
<code>
<xmp><% form.get("foo").render("<td>", "<br></td>"); %></xmp>
</code>
</blockquote>

@param	fd		the current form data object (if any). Form data objects 
				hold the results of submitted form data. This might be
				<tt>null</tt> initially (when the form is show to the user
				for the first time).
@param	writer 	the output destination
@param	prefix	prefix value for each element. specify
				<tt>null</tt> or an empty string if not needed.
@param	suffix	suffix value for each element. specify
				<tt>null</tt> or an empty string if not needed.
*/
public void render(FormData fd, Writer writer, String prefix, String suffix) 
throws SQLException, IOException
	{
	if (prefix != null)
		writer.write(prefix);
	render(fd, writer);
	if (suffix != null)
		writer.write(suffix);
	}

/**
Calls {@link renderError(Writer, String)} with &lt;br&gt; as
the message seperator string.
**/
public Field renderError(FormData fd, Writer writer) throws IOException
	{
	return renderError(fd, writer, "<br>\n"); 
	}

/**
Convenience method to render validation errors. Normally, a jsp would
always call {@link #getValidateErrors} and:
<ul>
	<li>if the returned data was null would write an empty
	string or nothing
	<li>the list of error messages adjacent to the html field
	itself.
</ul>
<p>
This method does exactly that: if applicable it writes validation error
messages, each seperated by the specfied message seperator parameter.
<p>
If this field is rendering style tags, then the error messages are
encapulated in a html &lt;span&gt; tag and the class of the span tag is set
to <tt>form-errmsg</tt>

@param	writer		the Writer to render errors to
@param	htmlSep		the seperator string (any string or html tag/fragment)
					to separate messages (if there are more than 1
					error messages)		

@return  this field object for method chaining convenience
**/
public Field renderError(FormData fd, Writer writer, String htmlSep) throws IOException 
	{
	List errors = getValidateErrors(fd);

	if (errors == null) {
		return this;
		}

	if (renderStyleTag) {
		writer.write("<span class='form-errmsg'>");
		}

	Iterator it = errors.iterator(); 
	while (it.hasNext()) {
		String item = (String) it.next();
		writer.write(item);
		writer.write(htmlSep);
		}	
	if (renderStyleTag) {
		writer.write("</span>");
		}
	return this;
	}

/**
Adds a field level validation error message. This method is needed when
some validation for this field is done via classes that are <b>not</b>
subclasses of {@link FieldValidator} (these classes are used outside of the
validation framework and their error message still needs to be added to the
field validation error data)
<p> 
<b> This is a very useful method for adding the results of arbitrary code
to this form. 
</b>

@see	Form#addError
**/
public void addError(FormData fd, String errorMessage)
	{
	List list = fd.createFieldErrorList(this);
	list.add(errorMessage);
	}

/**
Adds a refresher to this field. The refresher will be invoked to set
new values for this field before the field is rendered. 
*/
public void add(FieldRefresher refresher)
	{
	this.refresher = refresher;
	refresher.field = this;
	}

/**
Adds any arbitrary string to the field, which is written as is when
the field is rendered. (style tags should be set via the {@link #setStyleTag}
however).
<p>
This method can be called as many times as needed. For example:
<blockquote>
<pre>
addString("tabindex=2");
addString("onclick='foo();return true;'");
</pre>
</blockquote>
or to set a arbitrary Javascript string:
<blockquote>
<pre>
addString("onMouseOver='bar()' onClick='foo()'");
</pre>
</blockquote>
*/
public void addString(String str)
	{
	this.arbitraryString.add(str);
	}

/**
Adds some arbitrary text to this field which can later be retrieved via the
getLabel method and shown as an html label for this field. This is useful
when programmatically creating fields from a database where the label
itself is retrieved from the database.
*/
public void addLabel(String label) {
	this.label = label;
	}

/**
Returns the previously set label or <tt>null</tt> if no
label was set.
*/
public String getLabel() {
	return label;
	}

/** 
Validates this field via the installed validators. This method should be
called after setting the submitted value of this field via the {@link
#setValueFromSubmit} method.
<p>
No validators are run if the field is not enabled.
<p>
Validation errors that result after calling this method (if any)
can be retrieved via the {@link #getValidateErrors} method.

@return <tt>false</tt> on a validation error, <tt>true</tt> otherwise.
**/ 
public boolean validate(FormData fd, HttpServletRequest req) 
	{
	//fast return;
	int size = validators.size();
	if ( size == 0 )
		return true;
		
	boolean result = true;
	List errors = null;
	
	for (int n = 0; n < size; n++) 
		{
		FieldValidator v = (FieldValidator) validators.get(n);
				
		if (! v.validate(fd, req)) {
			result = false; //latch
			String tmp = v.getErrorMessage();
			log.bug("Field validate error: ", name, tmp);		
			errors = fd.createFieldErrorList(this);
			errors.add(tmp);
			}
		}	

	return result;	
	}

/** 
Returns a List of Strings containing validation errors. Items in this list
are a collection of all error messages (Strings) returned by validators
that validate this field.

If there are no validation errors, the returned list will be null.
**/
public List getValidateErrors(FormData fd) 
	{
	if (fd == null)
		return null;
		
	List list = fd.getFieldErrors(this); //can be null
	
	return list;
	}

/** 
Adds a validator for this field. During validation, validators are
sequentially called in the order they were added by this method. Note, this
method is package-private since it need only be called from a {@link
FieldValidator} constructor.
**/
void addValidator(FieldValidator validator) 
	{	
	validators.add(validator);
	}

List getValidators()
	{
	return validators;
	}

/* 
Returns <tt>true</tt> is this form element is required, <tt>false</tt>
otherwise. Required means that the value of this field cannot be null.
*/
/* No need -- use a validator
public boolean isRequired() 
	{
	return required;
	}
*/

/*
Specify <tt>true</tt> if this form element is required,
<tt>false</tt> otherwise. By default, fields are required.
*/
/*No need -- use a validator
public void setRequired(boolean val) 
	{
	required = val;
	}
*/

/** 
Enables this field for this particular request. Enabled fields run
through all attached validators. Non enabled fields skip all validation.
<p>
Note, since we use the {@link FormData} to keep track of enable/disable
status, fields can be enabled/disabled per user (independent of other
users).
**/
public void enable(FormData fd) 
	{
	if (fd != null)
		fd.enableField(name, true);
	}

/** 
Disabled this field for this particular request. Disabled fields skip all 
validation.
<p>
Note, since we use the {@link FormData} to keep track of enable/disable
status, fields can be enabled/disabled per user (indepenent of other
users).
**/
public void disable(FormData fd) 
	{
	if (fd != null)
		fd.enableField(name, false);
	}

/**
Enables this field for all users.
*/
public void enable()
	{
	enabled = true;
	}

/**
Disables this field for all users.
*/
public void disable()
	{
	enabled = false;
	}

public boolean isEnabled(FormData fd) 
	{
	if (fd == null)
		return true;
		
	return fd.isEnabled(name);
	}

/** 
Sets the parent form for this field. This method is automatically called by
a form object when this field is added to that form. This saves us from
having to pass in a form reference in the constructor for this field object
(which can be a hassle when constructing many field objects). Visibility is
package-private since this method need only be called from the base form
class.
**/
void setParentForm(Form form) 
	{
	Argcheck.notnull(form, "form parameter was null");
	this.form = form;
	}

/**
Sets a style tag to be rendered (overrides the default class tag of
<tt>formName-inputName</tt>.
*/
public void setStyleTag(String value) 
	{
	styleTag = value;
	}

/**
Set to <tt>true</tt> if the element should render a style tag, <tt>false</tt>
otherwise. By default this value is <tt>true</tt>.
<p>
An arbitrary style class can be set for a field by calling the {@link
#setStyleTag} method. If a custom style class is <b>not</b> set via this
method, style tags are rendered as:

<blockquote><pre>
class=formName-inputName
</pre></blockquote>

So, for example, a input type of name <tt>myname</tt> contained in
the form <tt>myform</tt> will render it's style tag as the following
class:

<blockquote><xmp>
<input ..... class="myform-myname" ....>
</xmp></blockquote>
Note, <tt>.</tt> and <tt>_</tt> are not allowed in css class names
(for legacy reasons) but <tt>-</tt> <i>is</i> allowed.
**/
public void renderStyleTag(boolean val) 
	{
	renderStyleTag = val;
	}

/**
Logs an unexpected submission state, typical of a hacked input form
*/
protected void hacklert(HttpServletRequest req, String msg)
throws SubmitHackedException
	{
	form.handleSubmitHacked(req, msg);
	}
	
public String toString() 
	{
	StringBuffer buf = new StringBuffer(256);
	buf.append("Field name='").append(name);
	
	/*
	can be null: if field was added to the form then the form sets itself
	as the parent of that field. However, if the field was a hidden field
	created dynamically and added to the formdata object, then there is no
	parent form. this can then be null.
	*/				   	
	if (form != null) {
		buf.append("' (@form=").append(form.getName()).append(")");		
		}
		
	buf.append("; type='").append(getType()).append("']");
	if (refresher != null) {
		buf.append(" Attached refresher: ");
		buf.append(refresher);
		}
	return buf.toString();
	}	
	
/** 
Encapsulates different types of HTML form elements. Needed
because various subclasses return a specific type which is
then used internally inside the {@link #render} method (
each HTML form element must have a type for example,
<tt>&lt;input <b>type=radio</b>...&gt;</tt> or <tt>&lt;input
<b>type=text</b>....&gt;</tt> etc.)
**/	
public static class Type 
	{
	private String name;
	private Type(String name) {
		this.name = name;
		}
	public String getName() {
		return name;
		}

	public static final Type CHECKBOX = new Type("checkbox");
	public static final Type HIDDEN = new Type("hidden");
	public static final Type TEXT = new Type("text");
	public static final Type TEXTAREA = new Type("textarea");
	public static final Type PASSWORD = new Type("password");
	//no separate radiogroup since no such thing in HTML
	//radiogroups = radio's with same name
	public static final Type RADIO = new Type("radio");
	public static final Type SELECT = new Type("select");

	public String toString() {
		return name;
		}
	}	//~ class Type
	
}          //~class Field