// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/** 
Represents a form field validator. Various subclasses
implement concrete code (via {@link #validate}) for various
validation strategies. Subclasses are typically specific to
both the field type and the type of validation. So for
example, the {@link ValidatorText} would verify that the
specified HTML form element of type "text" has the specified
length.
<p>
Field level validators differ from {@link FormValidator}
because validation is at the field (per field) level and is
independent of other fields in the form.
<p>
<b>Note</b>: Validator objects have state and a particular
instance of a validator should only be assigned to one form
field. To ensure this, validators take a field reference in
their constructors and automatically add themselves to the
specified field at construction time. During validation,
validators are sequentially called in the order they were
added (which implies in the order they were constructed).
<p>
<b>Thread safety:</b> None of the validation classes (like
all other form-related classes) are thread safe.
Synchronization must be done at a higher level, typically
the session level.

@author hursh jain
**/
public abstract class FieldValidator 
{
protected String 	errorMessage;
protected Field		field;
//protected boolean	isOptional;

/** 
Creates a new validator. 

@param	field			the field to validate. <b>This validator
						is added to the field automatically.
						Subclasses can later retrieve the field
						for validation</b>.
@param	errorMessage	the error message associated with invalid data
						See {@link #getErrorMessage}
**/
public FieldValidator(Field field, String errorMessage)
	{
	field.addValidator(this);
	this.errorMessage = errorMessage;
	this.field = field;
	}

/**
Validates the field in some fashion. 
<p>
If there are validation error, stores the error in the formdata 
and returns <tt>false</tt>, otherwise returns <tt>true</tt>
*/
public abstract boolean validate(FormData fd, HttpServletRequest req);

/*
Makes this an optional validator for this field. During
validation, optional validators attached to a field are invoked
if and only if optional validation for that field is enabled.
Optional validation is intended for cases where validation should
only proceed based on certain states of the form (for example,
certain choices in the form might imply that certain other parts
of the form must be validated and those same parts must
<b>not</b> be validated in other cases.
*/
/* >>>>>>>>>>>>> disable/enable the field instead.
public abstract setOptional(boolean val) {
	this.isOptional = true;
	}
*/

public String getErrorMessage() {
	return errorMessage;
	}

/**
Returns the field associated with this validator. 
*/
public Field getField() {
	return field;
	}
}          //~class FieldValidator