// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/** 
Represents an HTML hidden field. If this object is created with an array of
string values, then each string value will be written out as a seperate
hidden field and all such written hidden fields will have the same name.
When the form is submitted, all values can be retrieved as an array of
strings.

@author hursh jain
**/
public final class Hidden extends Field
{
static String[] empty_arr = new String[] { "" };

final static class Data {
	String[] values;
	}
	
String[] orig_values;
boolean	 dynamicallyAdded = false;

/** 
Creates a new text element with the initial value set to an empty string.
This is useful in cases where we are interested in the presense/absense of
the name of the hidden field and don't care about a seperate value.
**/
public Hidden(String name)
	{
	this(name, "");
	}
	
/** 
Creates a new hidden element with the specified initial value.
If the initial value is specified as <tt>null</tt>, it is set to be
an empty string.
**/
public Hidden(String name, String value)
	{
	super(name);
	if (value == null) 
		value = "";

	orig_values = new String[] { value };
	}

/** 
Convenience constructor to creates a new hidden element with the 
specified initial integer value (converted to a string).
**/
public Hidden(String name, int value)
	{
	super(name);
	orig_values = new String[] { String.valueOf(value) };
	}
	
/** 
Creates a new hidden element with the specified initial values.
**/
public Hidden(String name, String[] values) 
	{
	super(name);
	if (values == null) {
		log.warn("Specified values were null, defaulting to \"\"");
		orig_values = empty_arr;
		}
	else
		orig_values = values;
	}

/** 
Creates a new hidden element with the specified initial values. Each
element in the List will be added by invoking <tt>toString</tt> on it.
**/
public Hidden(String name, final List values) 
	{
	super(name);
	if (values == null) {
		log.warn("Specified values were null, defaulting to \"\"");
		orig_values = empty_arr;
		}
	else{
		//was: orig_values = (String[]) values.toArray(new String[0]);
		//this is more foolproof:
		final int size = values.size();
		orig_values = new String[size];
		for (int n = 0; n < size; n++) {
			orig_values[n] = values.get(n).toString();
			}
		}
	}

public Field.Type getType() {
	return Field.Type.HIDDEN;
	}

public void renderImpl(FormData fd, Writer writer) throws IOException 
 	{
 	/*
	We maintain the submit state even for hidden fields because it is 
	possible that some javascript on the client side might modify the 
	contents of a hidden field.
	*/
	String[] values = getRenderValues(fd);
	
	for (int n = 0; n < values.length; n++) 
		{
		writer.write("<input type='");
		writer.write(getType().toString());
		writer.write("' name='");
		writer.write(name);
		writer.write("'");
		
		if (values != null) {
			writer.write(" value='");
			writer.write(values[n]);
			writer.write("'"); 
			}

		final int arlen = arbitraryString.size();
		for (int k = 0; k < arlen; k++) {
			writer.write(" ");
			writer.write(arbitraryString.get(k).toString());
			}
	
		writer.write("></input>\n");	
		}
	}
			
/** 
Returns a String representing the value of this field or <tt>null</tt>
if there is no current value. If more than one value exists for this
hidden field, use the {@link getValues} method. If called when more
than one value exists, this method will return any 1 arbitrary value.
*/
public String getValue(FormData fd) 
	{
	if (dynamicallyAdded) {
		return orig_values[0];
		}
		
	Data data = (Data) fd.getData(name);

	if (data == null)
		return null;
			
	return data.values[0];
	}

/**
Returns a String[] containing all values associated with this field.
*/
public String[] getValues(FormData fd) 
	{
	if (dynamicallyAdded)
		return orig_values;

	Data data = (Data) fd.getData(name);
	
	if (data == null)
		return null;
			
	return data.values;
	}


/**
Convenience method that returns one value of this field as a String with
null values being returned as an empty string). <b>If more than one value
exists, use the {@link #getValues} method instead.</b>

@throws NumberFormatException	if the value could not be
								returned as in integer.	
*/
public String getStringValue(FormData fd)
	{
	String  s = getValue(fd);
	if (s == null)
		s = "";
	return s;
	}

/**
Convenience method that returns one value of this field as a Integer. <b>If
more than one value exists, use the {@link #getValues} method instead.</b>

@throws NumberFormatException	if the value could not be
								returned as an integer.	
*/
public int getIntValue(FormData fd) {
	String value = getValue(fd);
	if (value != null)
		value = value.trim();
	return Integer.parseInt(value);
	}

/**
Convenience method that returns one value of this field as a Short. <b>If
more than one value exists, use the {@link #getValues} method instead.</b>

@throws NumberFormatException	if the value could not be
								returned as a short.	
*/
public short getShortValue(FormData fd) {
	String value = getValue(fd);
	if (value != null)
		value = value.trim();
	return Short.parseShort(value);
	}

/**
Convenience method that returns one value of this field as a boolean. The
value is converted into a boolean as per the {@link
Boolean.valueOf(String)} method. <b>If more than one value exists, use the
{@link #getValues} method instead.</b>
*/
public boolean getBooleanValue(FormData fd) {
	return Boolean.valueOf(getValue(fd)).booleanValue();
	}

/**
The values to render this field with. If the form has not been shown to the
user at all or if the specified fd object is null, returns the original
values. Also if this field was dynamically added to the formdata, the
original values will be returned (since there will never be any submitted
values).
*/
String[] getRenderValues(FormData fd) 
	{
	if (dynamicallyAdded)
		return orig_values;
	
	if (fd != null) {
		Data data = (Data) fd.getData(name);	
		if (data == null)
			return empty_arr;
		else	
			return data.values;
		}
	else { //fd == null, no form data, showing form for first time
		return orig_values;
		}
	}

public void setValueFromSubmit(FormData fd, HttpServletRequest req) 
throws SubmitHackedException
	{
	String[] values = req.getParameterValues(name);
	
	if (values == null) 
		{  //client was buggy or hacked  [or field = disabled]
		if (! enabled || ! isEnabled(fd)) {  
			hacklert(req, "Bug/hack alert: did not find ["+name+"] field in the request (but expected to), defaulting to original values");
			}
		values = empty_arr;
		}
			
	Data data = new Data();
	fd.putData(name, data);
	data.values = values;
	}

/** 
Sets the <b>initial</b> values of this text element.
**/
public void setValue(String[] values) 
	{
	if (values == null) {
		log.warn("specified values param was null, defaulting to \"\"");
		values = empty_arr;
		}
		
	this.orig_values = values;
	}	

/** 
Sets the <b>initial</b> values of this text element.
**/
public void setValue(String value) 
	{
	if (value == null) {
		log.warn("specified values param was null, defaulting to \"\"");
		value = "";
		}
	
	this.orig_values = new String[] { value };
	}	

public boolean isFilled(FormData fd) 
	{
	Data data = (Data) fd.getData(name);
	
	if (data == null)
		return false;
		
	String[] values = data.values;
	
	if (values == null) {
		log.error("Internal error: unexpected state");
		}
	
	//since it's always at least some string -- possibly
	//all spaces but it'll be non-null
	return true; 
	}

public void reset(FormData fd) {
	Data data = (Data) fd.getData(name);
	data.values = orig_values;
	}

public String toString() 
	{
	return super.toString() + "; Orig. value: [" + Arrays.toString(orig_values) + "]"; 
	}	

}          //~class Hidden