// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.*;
import java.util.regex.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/**
A HTML TextArea
*/
public final class TextArea extends AbstractText
{
private WrapType wrap;
private int 	 rows = 0;
private int 	 cols = 0;
//private String	 classname = getClass().getName();

/** 
Creates a new textarea element with a initial value of "" (the
empty string). No initial rows/cols are specified which will have
the effect of rendering this field with the browsers default.
**/
public TextArea(String name)
	{
	this(name, "", 0, 0);
	}

/** 
Creates a new textarea element with a initial value of "" and
the specified rows/cols.
**/
public TextArea(String name, int rows, int cols)
	{
	this(name, "", rows, cols);
	}
	
/**
Creates a new text element with the specified initial value.
If the specified value is <tt>null</tt>, then the initial
value is set to "" (the empty string)
**/
public TextArea(String name, String value)
	{
	this(name, value, 0, 0);
	}

/**
Creates a new text element with the specified initial value
and rows/cols.
**/
public TextArea(String name, String value, int rows, int cols)
	{
	super(name, value);
	this.rows = rows;
	this.cols = cols;
	}

public Field.Type getType() {
	return Field.Type.TEXTAREA;
	}
	
public void renderImpl(FormData fd, Writer writer) throws IOException 
	{
	String value = getRenderValue(fd);
	
	writer.write("<textarea ");

	if (! enabled || ! isEnabled(fd)) {
		writer.write(" disabled");
		}
	
	writer.write(" name='");
	writer.write(name);
	writer.write("'");

	if (rows > 0) {
		writer.write(" rows='");
		
		// this is tricky, Writer != PrintWriter
		// and a int value will be treated as a character code
		// so writer.write(10) will write a newline, not '10'
		// writer.write(rows);  
		writer.write(String.valueOf(rows));
		writer.write("'"); 
		}
	
	if (cols > 0) {
		writer.write(" cols='");
		writer.write(String.valueOf(cols));
		writer.write("'"); 
		}

	if (wrap != null) {
		writer.write(" wrap='");
		writer.write(wrap.type);
		writer.write("'"); 
		}
			
	if (renderStyleTag) {
		writer.write(" style='");
		writer.write(styleTag);
		writer.write("'");
		}
		
	final int arlen = arbitraryString.size();
	for (int n = 0; n < arlen; n++) {
		writer.write(" ");
		writer.write(arbitraryString.get(n).toString());
		}

	writer.write(">");
	writer.write(sanitizedValue(value));
	writer.write("</textarea>");
	}
				
	
/** 
This value (if set) is rendered as the html <tt>COLS</tt> tag.

@return	this object for method chaining convenience
**/
public TextArea setCols(int cols) {
	this.cols = cols;
	return this;
	}	
			
/** 
This value (if set) is rendered as the html <tt>ROWS</tt> tag.

@return	this object for method chaining convenience
**/
public TextArea setRows(int rows) {
	this.rows = rows;
	return this;
	}	

/** 
Sets the wrapping mode of this text box. This value (if set) is
rendered as the html <tt>WRAP</tt> tag. By default the wrap
value is not set (and hence not rendered).

@return	this object for method chaining convenience
**/
public TextArea setWrap(WrapType wrap) {
	this.wrap = wrap;
	return this;
	}

//we need to sanitize our value since if we set to "</textarea>"
//the browser will think the value indicates the end of the
//textarea
static final Pattern pat = Pattern.compile("(?i)</textarea>");
private String sanitizedValue(String value) 
	{
	if ( value == null || value.equals("") ) {
		return "";		 
		}
		
	Matcher matcher = pat.matcher(value);
	String newval = matcher.replaceAll("&lt;/textarea&gt;");
	return newval;
	}
		
/** 
From the HTML spec:
<br>
<ol>
	<li>OFF disables word wrap. Text the user types is
	displayed with the exact line breaks that the user types.
	If the user explicitly inserts a line break, however, the
	break is included as part of the text area's value. The
	user has to scroll horizontally to see the ends of lines
	that do not fit in the text area element.
	<li>HARD causes word wrap, and the line breaks are
	included when the form is submitted. The text wraps inside
	the text area element, and that the user does not need to
	scroll horizontally.
	<li>SOFT causes word wrap, but the line breaks are not
	included when the form is submitted
</ol>
**/		
public static class WrapType {
	private String type;
	private WrapType(String type) { this.type = type; }
  	public static final WrapType OFF = new WrapType ("off");
	public static final WrapType HARD = new WrapType ("hard");
	public static final WrapType SOFT = new WrapType ("soft");
	}		
		
}          //~class TextArea