// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.*;
import java.util.regex.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/**
Validates that certain conditions in one or more fields in the form
require other conditions of some other fields in that form.
<p>
Since the pre/post conditions can be arbitrary, this class defines 
abstract methods called {@link #preConditionMet} and {@link
#postConditionMet}. If the pre condition is true, the post condition must
also be true. Subclasses should implement both methods as appropriate.

@author hursh jain
**/
public abstract class VConditional extends FormValidator
{
public VConditional(Form f, String name, String errorMessage)
	{
	super(f, name, errorMessage);
	}

/**
Returns the value returned by {@link #postConditionMet}
<i>if</i> the pre conditions were met. ({@link
#preConditionMet} returned true). If the pre condition was
<i>not</i> met, there is no need for further validation and
this method will return true.
*/
public boolean validate(FormData fd, HttpServletRequest req) 
	{
	if (! preConditionMet(fd, req) )
		return true;
		
	return postConditionMet(fd, req);
	}	


/**
Subclasses should implement this method to check that certain fields or
pre-conditions have been met. Only gf these conditions have been met will
there be a check to see that the post conditions are also true.

@return true if the pre-conditions have been met, false otherwise
*/
public abstract boolean preConditionMet(FormData fd, HttpServletRequest req);


/**
Subclasses should implement this method to check that 
certain post conditions have been met. This method will
only be called if the {@link #preConditionMet} method 
returns true.

@return true if the post-conditions have been met, false otherwise
*/
public abstract boolean postConditionMet(FormData fd, HttpServletRequest req);

}          //~class VConditional

