// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.*;
import java.util.regex.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/**
Allows for various types of text validation. This class is meant to be
used with text based input types such as subclasses of {@link
AbstractText}
**/
public final class VText extends FieldValidator
{
final static boolean dbg = false; //internal debugging

boolean 	checkLength = false;
boolean 	checkUnallowedChars = false;
boolean 	checkAllowedChars = false;
boolean 	checkAllowedPat = false;
boolean 	allowEmpty = true;  
boolean		trim = false;
//for a better toString()
boolean		intOnly = false;
boolean		floatOnly = false;
//

int 		minlength;
int 		maxlength;
String 		unallowedChars;
String 		allowedChars;
Pattern 	allowedPat;

/**  
Creates a new validator that by default only fails validation if the
field's value is empty, that is to say, it's not filled by the user or is
filled only with spaces (since spaces are removed before validation). This
can be changed via the {@link #trimSpaces} method.
<p>
Other methods in this class can be invoked for further kinds of
validation.
**/
public VText(AbstractText field, String errorMessage)
	{
	super(field, errorMessage);
	}

/**
If set to <tt>true</tt>, trims the string entered by the user before
attempting to run further validation on it. Defaults to <tt>false</tt>.
[Note: by default, AbstractText fields trim their values anyway so this
method is kinda redundant]
*/
public void trimSpaces(boolean val)
	{
	trim = val;
	}

/** 
Checks to see if the field is required to be filled by the user. If the
field is <b>allowed</b> to be empty (not filled) and <b>is</b> empty, then
validation succeeds and no further validation checks are done.

<p>This is useful, for example, if a field is required to be either
totally empty or filled with some sort of pattern as specified by {@link
#setAllowedPattern}. Note, however, that this method can get confusing if
a {@link VFilled} validator is also attached to this field (which can be
done automatically by database objects generated by {@link
fc.jdbc.dbo.Generate}. In that case, 2 error messages will be shown to the
user if the field is left blank - one for the attached VFilled validator
and one for this one.

@param	allow	<tt>true</tt> to allow for an empty field
				<tt>false</tt> otherwise (defaults to <tt>
				true</tt>).
				
@return	this object for method chaining convenience
**/
public VText allowEmpty(boolean allow) 
	{
	allowEmpty = allow;	
	return this;
	}

private static final String intOnlyRE 	= "\\s*\\d*\\s*";
private static final String floatOnlyRE	= "\\s*-?\\d*\\.?\\d*\\s*";

/**
Ensures that the string is composed of only integer characters, with
optional leading/trailing blanks. This is a convenience method that sets
the pattern to be <tt>"\s*\d*\s*"</tt>
*/
public VText allowIntegersOnly() {
	setAllowedPattern(Pattern.compile(intOnlyRE));
	intOnly = true;
	return this;
	}


/**
Ensures that the string is composed of only floating point number
characters with optional leading/trailing blanks. This is a convenience
method that sets the pattern to be <tt>"\s*-?\d*\.?\d*\s*"</tt>
*/
public VText allowFloatingOnly() {
	setAllowedPattern(Pattern.compile(floatOnlyRE));
	floatOnly = true;
	return this;
	}

/** 
Checks to see if the number of chars in the field are between the minimum
and maximum amount (both inclusive). If the minimum and maximum amounts
are the same (including 0), then the field has to be exactly that length.
(empty fields are also allowed if set via {@link #allowEmpty} method.

@return	this object for method chaining convenience
**/
public VText setLengthRange(int minlength, int maxlength) 
	{
	this.minlength = minlength;
	this.maxlength = maxlength;
	checkLength = true;
	return this;
	}

/** 
Checks to see if the number of chars in the field are between 0 and the
specified maximum amount (inclusive). [this method calls {@link
setLengthRange(0, maxlength)}]

@return	this object for method chaining convenience
**/
public VText setMaxSize(int maxlength) 
	{
	return setLengthRange(0, maxlength);	
	}

/** 
Checks to see if the number of chars in the field are at least
the minimum number (inclusive) specified by this method. 

@return	this object for method chaining convenience
**/
public VText setMinSize(int minlength) 
	{
	return setLengthRange(minlength, Integer.MAX_VALUE);	
	}

	
/** 
Sets the characters <b>not</b> allowed in this field. If some character is
marked as both allowed (via the {@link setAllowedChars} method) and
unallowed (via this method), then unallowed characters have precedence and
if found in the input, the field will not be validated.
<p>
The same functionality can be achieved via regular expressions and negated
character classes. It's a matter of preference.
<p>
After this method is called, the pattern previously set (if any) via the
{@link #setAllowedPattern} method will be ignored for validation.

@param	chars	the unallowed chars. This parameter must not be null.
@return	this object for method chaining convenience
**/	
public VText setUnallowedChars(String chars) 
	{
	Argcheck.notnull(chars, "specified param 'chars' was null");
	unallowedChars = chars;
	checkUnallowedChars = true;
	checkAllowedPat = false;
	return this;
	}

/** 
Sets the characters allowed in this field. All characters in the specified
string will be allowed, all else disallowed. An empty string (no
characters at all) is allowed at validation time if set via {@link
#allowEmpty} method.
<p>
After this method is called, the pattern previously set (if any) via the
{@link #setAllowedPattern} method will be ignored for validation.

@param	chars	the allowed chars. This parameter must not be null.
@return	this object for method chaining convenience
**/	
public VText setAllowedChars(String chars) 
	{
	Argcheck.notnull(chars, "specified param 'chars' was null");
	allowedChars = chars;
	checkAllowedChars = true;
	checkAllowedPat = false;
	return this;
	}

/** 
Sets the regular expression representing the allowed input. The pattern
will be matched with the entire value of the field.
<p>
After this method is called, the string previously set (if any) via the
{@link #setAllowedChars} method will be ignored for validation.

@param	pat		the allowed pattern. Must not be null.
@return	this object for method chaining convenience
**/	 
public VText setAllowedPattern(Pattern pat) 
	{
	Argcheck.notnull(pat, "specified param 'pat' was null");
	allowedPat = pat;
	checkAllowedPat = true;
	checkAllowedChars = false;
	
	//callers should set these back to true
	intOnly = false;
	floatOnly = false;
	
	return this;
	}

public String toString()
	{
	StringBuffer buf = new StringBuffer(512);
	buf.append("VText(");
	if (checkLength) {
		buf.append("Len=[");
		buf.append(minlength);
		buf.append("-");
		buf.append(maxlength);
		buf.append("]");
		buf.append(", ");
		}
	if (checkUnallowedChars) {
		buf.append("Unallowed=");
		buf.append(unallowedChars);
		buf.append(", ");
		}
	if (checkAllowedChars) {
		buf.append("Allowed=");
		buf.append(allowedChars);
		buf.append(", ");
		}	
	if (checkAllowedPat) 
		{
		if (intOnly) {
			buf.append("ints only");
			}
		else if (floatOnly) {
			buf.append("floats only");
			}
		else {
			buf.append("Pattern=");
			buf.append(StringUtil.viewableAscii(allowedPat.pattern()));
			}
		buf.append(", ");
		}
	if (trim) {
		buf.append("Trim=true");
		buf.append(", ");
		}

	buf.deleteCharAt(buf.length() -1);
	buf.deleteCharAt(buf.length() -1);
	buf.append(")");
	
	return buf.toString();
	}

		
public boolean validate(FormData fd, HttpServletRequest req) 
	{
	String val = ((AbstractText)field).getValue(fd);
	
	/*
	can be null if not submitted, not filled by user or even
	initially if some field was constructed with a default
	value of null
	*/
	if (val == null) 
		{ 
		if (dbg) System.out.println(">>>>>>>>>> val == null, returning false");
		
		if (allowEmpty)
			return true;
		else
			return false;
		}
	
	if (trim)
		val = val.trim();

	//check for empty again after trimming.
	if (allowEmpty && isEmpty(val)) {
		if (dbg) System.out.println("RETURNING TRUE");
		return true;
		}
	
	if ( ! allowEmpty && isEmpty(val)) {
		if (dbg) System.out.println("RETURNING FALSE");
		return false;
		}
	
	boolean result = true;
	if (dbg) System.out.println(">>>>>>>>>>> ALLOW_EMPTY=" + allowEmpty + "; isEmpty()=" + isEmpty(val) + "; VALUE=[" + val + "]");

	// keep going thru all other checks to see if there is an error
			
	if (checkLength) { //checking for length ?
		if (result) { //latch, test further since result currently true
			result = checkLength(val); 
			}
		}

	if (checkUnallowedChars) { //checking for unallowed chars ?
		if (result) { //latch, test further since result currently true
			result = checkUnallowedChars(val);
			}
		}
	
	if (checkAllowedChars) { //checking for allowed chars ?
		if (result) { //latch, test further since result currently true
			result = checkAllowedChars(val);
			}
		}

	if (checkAllowedPat) { //checking for regex pattern ?
		if (result) { //latch, test further since result currently true
			result = checkAllowedPat(val);
			}
		}
		
	return result;
	}	

final boolean isEmpty(String str) 
	{
	if (str.length() == 0) {
		return true;
		}
	
	return false;
	}

final boolean checkLength(String str) 
	{
	if (str == null) {
		return false;
		}

	int len = str.length();
	if (len < minlength || len > maxlength) {
		return false;
		}

	return true;
	}

final boolean checkAllowedChars(String str)
	{
	if (str == null)
		return false;
				
	boolean found = false;
	for (int n = 0; n < str.length(); n++) 
		{
		char c = str.charAt(n);
		for (int k = 0; k < allowedChars.length(); k++) 
			{
			char mchar = allowedChars.charAt(k);
			if (c == mchar) {
				found = true;
				break;
				}
			}
		}
	return found;
	}


final boolean checkUnallowedChars(String str)
	{
	if (str == null)
		return false;
				
	boolean found = false;
	for (int n = 0; n < str.length(); n++) 
		{
		char c = str.charAt(n);
		for (int k = 0; k < unallowedChars.length(); k++) 
			{
			char mchar = unallowedChars.charAt(k);
			if (c == mchar) {
				found = true;
				break;
				}
			}
		}

	if (found) {		//found an unallowed char
		return false;
		}
	else
		return true;
	}

final boolean checkAllowedPat(String str)
	{
	if (str == null)
		return false;
	
	Matcher matcher = allowedPat.matcher(str);

	return matcher.matches();
	}		

}          //~class VTextLength

