// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.forms;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/** 
A dependent choice group that can change it's data depending on what
the user has selected in other parts of the form.

@author hursh jain
**/
public abstract class RefreshableChoiceGroup extends Field
{
public static class Data
	{
	//------------- choice data ----------------------
	//choices in this group, linked hash map maintains insertion order
	private	Map		options  = new LinkedHashMap();		  
	//we need this to retrieve options by index (0..n) etc
	//(the options Map being a linked hash map gives us an
	//iteration in insertion order but not by any index.
	private List	optionsList = new ArrayList();		  
	//orig selected options
	private Map		origSelectedMap =  new HashMap();  
	//if true, options will be rendered in reverse
	private boolean reverseRender = false;
	private	Map		reverseOptions;	 //lazy creation		
	//------------- submit data -----------------------------
	//options submitted by user [Choice.getValue()->Choice]
	Map		selectedMap = null;  //created if present on submit	 
	//--------------------------------------------------------
	
	public void addChoice(ChoiceGroup.Choice choice) 
		{
		String val = choice.getValue();
		options.put(val, choice);
		optionsList.add(choice);
		if (choice.isOrigSelected()) {
			origSelectedMap.put(val, choice);	
			}	
		//so this will be recreated with the new option when
		//reverseRender is called again
		reverseOptions = null;
		choice.parent = DependentChoiceGroup.this;
		}
	}
		
/** 
Creates a new grouped choice object that intitially contains no choice
fields

@param 	name		the field name
**/
public DependentChoiceGroup(String name)
	{
	super(name);
	}

public abstract Field.Type getType();


/** 
Returns a Collection containing the choices selected by the user. Each
item in this collection is of type {@link ChoiceGroup.Choice}. If there
are no selected options, returns an empty unmodifiable collection.

@param	fd 	the submited form data
**/
public Collection getValue(FormData fd) 
	{
	DependentChoiceGroup.Data data = (DependentChoiceGroup.Data) fd.getData(name);
	if (data == null)
		return Form.empty_list;
	return data.selectedMap.values();
	}

/**
Sets the selected values for this choicegroup in the specified form data.
<u>The set of choices in the choicegroup are not set by this method, only
<i>which</i> of the choices that will be displayed as
selected/not-selected for the request associated with the form data.</u>
<p>
To set the choices themselves, use the appropriate constructor or call
the {@link setValue(Collection)} method. <b>.The specified form data must
not be null.</b>

@param	fd		the non-null form data object for rendering the form
@param	values	a collection of {@link ChoiceGroup.Choice} objects
*/
public void setValue(FormData fd, Collection values)
	{
	Argcheck.notnull(fd, "specified fd param was null");
	Argcheck.notnull(values, "specified values param was null");
		
	DependentChoiceGroup.Data data = new ChoiceGroup.Data();
	fd.putData(name, data);

	Iterator it = values.iterator();
	while (it.hasNext()) {
		ChoiceGroup.Choice choice = (Choice) it.next();
		if (choice.isOrigSelected()) {
			data.selectedMap.put(choice.getValue(), choice);	
			}
		}
	}

/**
Sets the values for this choice group. Any previously set values are
first cleared before new values are set.

@param	values	a collection of {@link ChoiceGroup.Choice} objects
*/
public void setValue(Collection values)
	{
	Argcheck.notnull(values, "specified values param was null");
	
	Iterator it = values.iterator();
	while (it.hasNext()) {
		ChoiceGroup.Choice choice = (Choice) it.next();
		add(choice);
		}
	}

/**
Returns the choice specified by the index n. The choices are indexed in
the order that they were <i>added</i> to the form. (regardless of whether
this field is being reverse rendered).

@throws IndexOutOfBoundsException 	if the specified index is out
									of range.
*/
public ChoiceGroup.Choice getChoice(int n)
	{
	return (ChoiceGroup.Choice) optionsList.get(n);	
	}

/**
Returns <tt>true</tt> if some option has been selected by the user,
<tt>false</tt> otherwise. (also returns <tt>false</tt> is the specified
form data is <tt>null</tt>).
*/
public boolean isFilled(FormData fd) 
	{
	if (fd == null)
		return false;
		
	ChoiceGroup.Data data = (ChoiceGroup.Data) fd.getData(name);
	return (data != null &&
			data.selectedMap.size() != 0);
	}

public void setValueFromSubmit(FormData fd, HttpServletRequest req) 
	{	
	String[] values = req.getParameterValues(name);
	
	/*
	can be null if a) not present in the html form or b) present
	in the form but not selected at all by the user.
	*/
	if (values == null) {
		return;
		}
	
	//instantiate only when needed
	ChoiceGroup.Data data = new ChoiceGroup.Data();	
	fd.putData(name, data);

	for (int n = 0; n < values.length; n++) 
		{
		//our options were stored with options' value as the key
    	Choice opt = (Choice) options.get(values[n]);
	
	// this can happen if option values were hacked by the client
    	if (opt == null) {
    		StringBuffer sb = new StringBuffer(512);
			sb.append(".setSubmittedValue(): could not match/retrieve a submitted choice from the optionMap")
    			.append("fieldname=").append(name)
    			.append("; submited value='") 
    			.append(values[n])
    			.append("'; choices=").append(options); 
			hacklert(req, sb.toString());
    		}				
		else {
			data.selectedMap.put(opt.getValue(), opt);
			}
		}
	}		//~setValueFromSubmit


/**
If called with <tt>true</tt>, renders the choices contained in this group
in the reverse order in which they were added (by default, choices are
rendered in the order they were added)
*/
public void reverseRender(boolean val)
	{
	synchronized (this) 
		{ //for mem vis.
		reverseRender = val;
		
		if (reverseRender && reverseOptions == null) 
			{
			reverseOptions = new LinkedHashMap();
			List list = new ArrayList();
			Iterator it = options.entrySet().iterator();
			while (it.hasNext()) {
				list.add(it.next());
				}
			int count = list.size() - 1;
			for (int n = count; n >= 0 ; n--) {
				Map.Entry e = (Map.Entry) list.get(n);
				reverseOptions.put(e.getKey(), e.getValue());
				}
			}
		}
	}

/**
Renders all elements of this group one after onether, seperated by a
whitespace. For more control over rendering spacing and direction, use the
{@link #render(FormData, Writer, String, String)} method. It's also
possible to render each item in a group individually by <b>not</b> calling
this method but getting a Map/List of elements in this group and rendering
them in whichever location desired. (by calling render on each specific
choice itself).
<p>
Specify <tt>null</tt> for the formdata if rendering this element for the
first time (before it has been submitted to by the user).
*/
public void renderImpl(FormData fd, Writer writer) throws IOException
	{
	renderImpl(fd, writer, null, "&nbsp;");
	}


/**
Renders the elements of this form by prefixing and suffixing each element
with the specified arguments.

@param	writer 	the output destination
@param	prefix	prefix value for each element. specify
				<tt>null</tt> or an empty string if not
				needed.
@param	suffix	suffix value for each element. specify
				<tt>null</tt> or an empty string if not
				needed.
*/
public void renderImpl(FormData fd, Writer writer, String prefix, String suffix) 
throws IOException
	{
	ChoiceGroup.Data data = null;

	if (fd != null) {
		data = (ChoiceGroup.Data) fd.getData(name);
		}

	Iterator it = null;
	if (reverseRender)
		it = reverseOptions.values().iterator();	
	else	
		it = options.values().iterator(); 
	
	while (it.hasNext()) 
		{
		Choice item = (Choice) it.next();
		String itemval = item.getValue();

		boolean selected = false;
		
		if (fd != null) 	/* maintain submit state */
			{					
			if (data != null) {  
				selected = data.selectedMap.containsKey(itemval);
				}
			else {  //form submitted but option not selected
				selected = false;
				}
			}
		else {  	
			/* show original state */
		  	selected = origSelectedMap.containsKey(itemval);
		   	}
	
		boolean disabled = ! isEnabled(fd);
	
		if (prefix != null)
			writer.write(prefix);
		item.render(writer, selected, disabled);
		if (suffix != null)
			writer.write(suffix);
		}	
	}

/**
Clears all values in this group.
*/
public void reset()
	{
	options.clear();
	optionsList.clear();		  
	origSelectedMap.clear();  
	reverseRender = false;
	if (reverseOptions != null)
		reverseOptions.clear();	
	}

/**
Returns a map containing {@link Choice} all elements contained within this
group. This is useful if the elements need to be rendered individually for
custom positioning on an html page. 
<p> 
Each elements in the map will reflect the original selection state when it
was created. Each element can optionally be checked to see if it was
selected by the user by calling the {@link #isSelected} method.
*/
public Map getAllElements() 
	{
	return options;
	}

/**
Returns <tt>true</tt> is the specified choice was selected by the user
(the user's submission is provided by the formdata argument).

@param	fd		form data submitted by user (can be null)
@param	choice	a choice option belonging to this group
@param	default_val	
				the default value to return in case the form
				data was null or did not contain this
				choicegroup. This will typically be
				<tt>false</tt> or
				<tt>choice.isOrigSelected()</tt>
*/
public boolean isSelected(
	FormData fd, ChoiceGroup.Choice choice, boolean default_val)
	{
	if (fd == null)
		return default_val;
		
	Argcheck.notnull(choice, "choice param was null");
	String itemval = choice.getValue();
	
	ChoiceGroup.Data data = 
			(ChoiceGroup.Data) fd.getData(name);

	if (data == null)
		return default_val;
		
	boolean selected = data.selectedMap.containsKey(itemval);
	return selected;
	}
	
/**
Utility methods that calls the {@link Choice#writeLabel} method
for each choice contained in this group
*/
public void writeLabel(boolean val)  
	{
	Iterator it = options.values().iterator();
	while (it.hasNext()) {
		ChoiceGroup.Choice item = (ChoiceGroup.Choice) it.next();
		item.writeLabel(val);
		}
	}

public String toString() 
	{	
	String linesep = IOUtil.LINE_SEP;
	StringBuffer buf = new StringBuffer(super.toString());
	buf.append("; Orig. values: ");
	buf.append(linesep);		

	Iterator it = null;
	if (reverseRender)
		it = reverseOptions.values().iterator();	
	else	
		it = options.values().iterator(); 
	
	while(it.hasNext()) {
		buf.append("\t");
		buf.append(it.next());
		buf.append(linesep);		
		}
	return buf.toString();	
	}	

/* 
No longer this way:

Creates a new choice in this choice group. This inner class is <b>not</b>
a static class but is a <b>member</b> class. Therefore to create new
instances of this class, the following syntax is used:
<blockquote>
<pre>
	mychoicegroup.new Choice(...);
</pre>
where <tt>mychoicegroup</tt> is some reference to a <tt>ChoiceGroup</tt>
object.
</blockquote>
Creating a new choice instance automatically adds it to the choice group
object.
*/

//WE DONT USE A NON-STATIC INNER CLASS ANYMORE (WHICH WE SHOULD BECAUSE
//BECAUSE EACH CHOICE NEEDS A REFERENCE TO IT'S PARENT) -- BECAUSE IT
//MAKES ADDING NEW CHOICES HARDER.

/**
Creates a new choice for this choice group. 

Also note that unlike other fields, the HTML label/text for this choice
has to be provided via the constructor (as opposed to being written at the
jsp level). See {@link #htmlBeforeField}.
**/
public static class Choice
	{
	private ChoiceGroup parent; //set by choicegroup when added to it
	private String 		value;
	private String 		label;
	private String 		labelsep = " ";
	private boolean 	orig_selected;
	private boolean		writeLabel = true;
	private boolean 	writeLabelAfter = true;
	private boolean 	writeLabelBefore = false;

	/** 
	Creates a new choice object.

	@param 	value		the value of this choice item
	@param 	label		the label (any html text) for this choice. 
	@param	selected	<tt>true</tt> is this choice is
						originally selected
	**/
	public Choice(String label, String value, boolean selected)
		{
		this.label = label;
		this.value = value;
		this.orig_selected = selected;
		//add(this);
		}

	/** 
	Constructs a new unselected choice with the specified value 
	and HTML text. 

	@param 	value		the value of this choice item
	@param 	label	the label (any html text) for this choice
	**/
	public Choice(String label, String value) {
		this(label, value, false);
		}

	/** 
	Constructs a new unselected choice with the specified label (and no
	separate value attribute)

	@param 	label	the label (any html) text for this choice
	@param	selected	<tt>true</tt> is this choice is
						originally selected
	**/
	public Choice(String label, boolean selected) {
		this(label, null, selected);
		}

	//makes a new copy and adds it to the specified 
	//radioGroup
	private void copyTo(ChoiceGroup target) {
		Choice c = new Choice(value, label, orig_selected);
		c.writeLabel = writeLabel;
		c.writeLabelAfter = writeLabelAfter;
		c.writeLabelBefore = writeLabelBefore;
		target.add(c);
		}
	
	/**  
	By default, the HTML label (if any) is written after the input element
	tag but calling this method reverses this order.
	**/
	public void labelBeforeField() {
		writeLabelAfter = false;
		writeLabelBefore = true;
		}

	/**
	Specify <tt>true</tt> to write the label for this choice,
	<tt>false</tt> to skip the label. By default, this is <tt>true</tt>.
	(false is useful when just the radio button need to be shown, say in a
	table row with the label shown in a seperate header row).
	*/
	public void writeLabel(boolean val) {
		this.writeLabel = val;
		}

	/**
	Renders this choice maintaining it's selected state by using the
	specified form data.
	<p> Each choice can be rendered separately which helps in arbitrary
	html layout. Choices can also be rendered together via the parent
	{@link ChoiceGroup#render(FormData, Writer} method.	
	*/
	public void render(FormData fd, Writer writer, boolean disabled)
	throws IOException	
		{
		ChoiceGroup.Data data = null;

		if (fd != null) {
			data = (ChoiceGroup.Data) fd.getData(parent.name);
			}

		boolean selected = false;
		
		if (fd != null) 	/* maintain submit state */
			{					
			if (data != null) {  
				selected = data.selectedMap.containsKey(getValue());
				}
			else {  //form submitted but option not selected
				selected = false;
				}
			}
		else {  	
			/* show original state */
		  	selected = isOrigSelected();
		   	}
	
		render(writer, selected, disabled);		
		}
		
	/**
	Renders this choice with the select state specified by the
	<tt>selected</tt> parameter.
	<p> Each choice can be rendered separately which helps in arbitrary
	html layout. Choices can also be rendered together via the parent
	{@link ChoiceGroup#render(FormData, Writer} method.
	*/	
	public void render(Writer writer, boolean selected, boolean disabled) 
	throws IOException
		{
		if (writeLabel && writeLabelBefore) {
			writer.write(label);
			writer.write(labelsep);
			}
	
		writer.write("<input type='");
		writer.write(parent.getType().toString()); //type of choice (outer class)
		writer.write("' name='");
		writer.write(parent.name);
		writer.write("'");
		
		if (value != null) {  //value tag present
			writer.write(" value='");
			writer.write(value);
			writer.write("'"); 
			}
	
		if (selected) {
			writer.write(" checked");
			}
		
		if (parent.renderStyleTag) {
			writer.write(" style='");
			writer.write(parent.styleTag);
			writer.write("'");
			}
		
		final int arlen = parent.arbitraryString.size();
		for (int n = 0; n < arlen; n++) {
			writer.write(" ");
			writer.write(parent.arbitraryString.get(n).toString());
			}

		writer.write(">");
			
		if (writeLabel && writeLabelAfter) {
			writer.write(labelsep);
			writer.write(label);
			}
		writer.write("</input>");
		}

	/** 
	Returns the value of this choice. If no value is set, returns the html
	text value for this choice tag.
	**/
	public String getValue() 
		{
		if (value != null)
			return value;
		else
			return label;
		}	
	
	/**
	Convenience method that returns the value of this 
	choice as a Integer. 
	
	@throws NumberFormatException	if the value could not be
									returned as an integer.	
	*/
	public int getIntValue() {
		return Integer.parseInt(getValue());
		}
	
	/**
	Convenience method that returns the value of this choice as a Short.
	
	@throws NumberFormatException	if the value could not be
									returned as a short.	
	*/
	public short getShortValue(FormData fd) {
		return Short.parseShort(getValue());
		}
	
	/**
	Convenience method that returns the value of this choice as a boolean.
	The value is converted into a boolean as per the {@link
	Boolean.valueOf(String)} method.
	*/
	public boolean getBooleanValue(FormData fd) {
		return Boolean.valueOf(getValue()).booleanValue();
		}
		
	/**
	Returns the label for this choice. 
	*/
	public String getLabel()
		{
		return label;
		}
	
	/**
	Sets the seperator between labels and the choice. Defaults to a space
	if not set.
	*/
	public void setLabelSeperator(String sep)
		{
		labelsep = sep;
		}
	
	/** 
	@return <tt>true</tt> if this field was originally set to 
	selected, <tt>false</tt> otherwise
	**/
	public boolean isOrigSelected() {
		return orig_selected;
		}

	public String toString() 
		{
		return "ChoiceGroup.Choice: [value=" + value + 
				"; label=" + label + 
		/*
		note, we don't want to call form.toString() because that would
		result in a recursive loop if form.toString() is ever changed to
		call toString() of all it's constituent fields
		*/	
				"]"; 
		//ok but don't need this: form= + form.getName();
		}
	
	}   //~innerclass Choice    		

}          