// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.page;

import java.io.*;
import java.util.*;

import javax.servlet.*;
import javax.servlet.http.*;

/** 
A page is a cool and sane replacement for JSP's with much better syntax.
<p>
Server side pages implement this interface via the concrete {@link PageImpl}
class. This class has some additional utility methods as well that page
authors may find useful.

@author hursh jain
*/
public interface Page
{
//all pages result in java files with this package name;
public static String 	PACKAGE_NAME = "molly.pages";

//not used: public static String 	DEFAULT_ERROR_PAGE 	= "/molly/error.mp";
public static int 		DEFAULT_BUFFER_SIZE 	= 8048;

/** 
The default mime-type for each page. Usually, there should be no reason
to change this. However, if need be, this can be changed via a page
compiler directive or turned off entirely. If it's turned off, then the
mime-type should be manually specified via the {@link
javax.servlet.ServletResponse.setContentType} method.
*/
public static String 	DEFAULT_MIME_TYPE 		= "text/html";

/** 
The default encoding of the page, specified in the header sent back to
the client. This can be changed to utf-8, utf-16 or any other coding by
a page directive. Alternately, this can be set to an empty string and a
different encoding can also be specified in the &lt;head&gt; section of
the html document, for example:
<pre>
&lt;head&gt;&lt;meta http-equiv=content-type content='text/html; charset=UTF-8'&gt;&lt;/head&gt;
</pre>
<font color=red>Encodings can be tricky. We are first compiling
a page into a java source file, then running the source file and
sending it's output to the browser. <b>Read the page 
<a href="http://www.mollypages.org/page/charset.mp">encoding
and charsets</a> if you are using any non-us-ascii or non-
ISO-8859-1</b> characters in your molly source page.</font>
*/
public static String 	DEFAULT_ENCODING 		= "ISO-8859-1";

public void render(HttpServletRequest req, HttpServletResponse res) throws Exception;

/**
This method is invoked whenever a page is created and before it
is run.
<p>
Pages should override this method to instantiate/set-up page
variables as needed. (pages have no-arg constructors so like
most of the servlet API, setup and initialization of variables
is done in a init method instead).
<p>
When overriding this class, you must remember to 
call: <tt>super.init</tt>
<p>
The page class is reloaded if the page is modified. Variables
should therefore be cleaned up in the {@link destory} method 
as needed.
*/
public void init(PageServlet servlet, String contextRelativePagePath) throws ServletException;

/**
This method is invoked whenever a page is destoryed/unloaded
*/
public void destroy();

/**
Returns the path to this page from the web servers <i>document root</i>. 
<p>So for example, if the page is at <code>foo/bar.mp</code> and is running
under the webapp context of <code>context1</code>, then the page path
will be: <code>/context1/foo/bar.mp</code>. If there is no specific
web app (i.e., the most common case of a default "" webapp), then the page
path will be <code>/foo/bar.mp</code>
<p>
This page path is essentially what needs to be typed in the browsers
URL window to invoke the page. It's also useful as form action parameters.
For example, in a molly page:
<blockquote>
<pre>
..
&lt;form action="[=getPagePath(req)]" method="post"&gt;
..
&lt;/form&gt;
</pre>
</blockquote>
This will submit the form to the same page where the form is defined. This
can be hard coded of course but by using <code>getPagePath</code>, the html
does not have to be changed if the name of the page changes on disk.
*/
public String getPagePath(HttpServletRequest req);

/**
Returns the real absolute directory path for the {@link #getPagePath PagePath}.
<p>
So, for example, for a webserver document root at
<code>/web/sites/default/</code> and a page located in
<code>foo/bar.mp</code>, the real path will be:
<code>/web/sites/default/foo/bar.mp</code>
*/
public String getRealPath(HttpServletRequest req);

/**
Redirects the client to the new page location. This is a thin (possibly
easier to remember) wrapper around the {@link HttpServletResponse.sendRedirect} method.
<p>
The location parameter can be relative to the specified request's URI or
relative to the context root if it contains a leading '/'. The webapp name
(if any) does <b>not</b> have to be specified, the redirect will creates a
full URL <u>(including the webapp context path)</u> suitable for this
purpose. 
<p>
For example:
<style>
#redirects {background: #cccccc; }
#redirects tr {background: white; }
#redirects .head {font-weight: bold; }
</style>
<table id=redirects border=0 cellspacing=1 cellpadding=7>
<tr class=head>
	<td width="25%">webapp context</td>
	<td width="25%">current page</td>
	<td width="25%">location parameter</td>
	<td width="25%">resulting page</td>
</tr>
<tr>
	<td>default web app ("/")</td>
	<td>foo/bar.mp</td>
	<td>baz.mp</td>
	<td>foo/baz.mp</td>
</tr>
<tr>
	<td>default web app ("/")</td>
	<td>foo/bar.mp</td>
	<td>/baz.mp</td>
	<td>baz.mp</td>
</tr>
<tr>
	<td>/myapp</td>
	<td>foo/bar.mp</td>
	<td>baz.mp</td>
	<td>/myapp/foo/baz.mp</td>
</tr>
<tr>
	<td>/myapp</td>
	<td>foo/bar.mp</td>
	<td>/baz.mp</td>
	<td>/myapp/baz.mp</td>
</tr>
</table>

@param	req			the current request
@param	res			the current response
@param	location	location to redirect to.
*/
public void clientRedirect(HttpServletRequest req, HttpServletResponse res, String newLocation) throws IOException;

/**
Returns a thread specific CharArrayWriter that can be passed to this method
as various points in the page. The contents of this writer can then be
printed on the page when desired. 
<p>
Note: The writer is <b>not</b> reset or flushed when it is retrieved. It
must be <font color=blue>reset manually</font> via calling the {@link
java.io.CharArrayWriter#reset} method. This design-decision allows request
threads to collect debugging data across multiple pages.
<p>
The suggested usage idiom is:
<blockquote>
<pre>
	dbg(true);
	CharArrayWriter <font color=blue>cw</font> = getThreadLocalWriter():
	bug(<font color=blue>cw</font>, "some message");
	...
	bug(<font color=blue>cw</font>, "other message");
	...
	<font color=blue>
	cw.writeTo(out);
	cw.reset();
	</font>
</pre>
</blockquote>
*/
public CharArrayWriter getThreadLocalWriter();

/* 
These are in PageImpl but do they need to be in the Page interface as well ?
Subject to change so prolly not, thus commented out. hj

public void startTimer();
public long getTime();
public void dbg(boolean val);
public void dbgPrefix(String dbg_prefix);
public void dbgSuffix(String dbg_suffix);
public void bug(final Writer writer, final Object str1) throws IOException;
public void bug(final Writer writer, final Object str1, final Object str2) throws IOException;
public void bug(final Writer writer, final Object str1, final Object str2, final Object str3) throws IOException;
public void bug(final Writer writer, final Object str1, final Object str2, final Object str3, final Object... args) throws IOException;
*/
}
