// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.page;

import java.io.*;
import java.util.*;
import javax.servlet.*;
import javax.servlet.http.*;

import fc.io.*;
import fc.web.servlet.*;

/** 
The page servlet. Handles web requests that are mapped to a molly server
page (typically via a <code>*.mp</code> extension).
<p>
All uncaught exceptions in the rendered page are wrapped in a
<tt>ServletException</tt> and thrown up to the servlet container. These are
typically handled by the container by either showing the full stack trace
to the user or using a error page configured in the containers
<tt>web.xml</tt> file. This latter approach is recommended for production
use. An
<i>example</i> is shown below.
	<blockquote>
	<xmp>
	<!-- in web.xml -->
	<error-page> 
		<exception-type>java.lang.Exception</exception-type> 
		<location>/errors/error.mp</location> 
	</error-page>
	</xmp>
	</blockquote>
<p>
However, if some partial response has already been sent to the browser and
an exception occurs (later in java code on that same page), then the
error page (if configured in web.xml) cannot typically be displayed.
<p>
For that scenario, this servlet accepts an <i>optional</i>
<b><tt>error_page</tt></b> initialization parameter. This parameter, if
present, should contain a <i>webapp</i>-relative path to an error page that
will be <i>included</i> in the rendered page if there is an exception
<i>after</i> the response has been committed. This error page is then
<i>included</i> in the response sent to the browser.
<p>
The following attributes are available in the error page
<pre>
javax.servlet.error.status_code 
javax.servlet.error.exception
javax.servlet.error.request_uri
javax.servlet.error.servlet_name
</pre>
<p>
This servlet also accepts an <i>optional</i> <b><tt>404_page</tt></b>
parameter. This parameter, if present, should contain a <i>web document
root</i>-relative path to a 404 or not found page. This page is different
than the <i>error_page</i> because it signifies a badly typed URL request
for a page that does not exist.  (for example,
<tt>http://somehost/badpage.mp</tt>). This parameter should be the same as
the 404 error code parameter in <tt>web.xml</tt>.
For example:
	<blockquote>
	<xmp>
	<!-- in web.xml -->
	<error-page> 
		<error-code>404</error-code> 
		<location>/errors/not_found.html</location> 
	</error-page> 
	</xmp>
	</blockquote>
<p>

@author hursh jain
*/
public class PageServlet extends FCBaseServlet
{
private static final boolean dbg = false;

PageMgr 	pagemgr;

//error page to include if the response has already been committed
//and an exception occurs.
String		error_page;
String		page_404;

public void init(ServletConfig conf) throws ServletException 
	{
	super.init(conf);
	try {
		ServletContext context = conf.getServletContext();

		synchronized (this)
			{	
			String docrootstr = context.getRealPath("/"); 
			/*
			docroot will be:
			context = "", path = <wwwroot>
			context = foo, path = <wwwroot>/foo
			*/
			if (docrootstr == null)
				throw new UnavailableException("docroot == null, so I ask you, what can I do ?");

			File docroot =  new File(docrootstr);			
			if (! docroot.exists())
				throw new UnavailableException("docroot " + docroot + " does not exist...so I ask you, what can I do ?");

			File scratchroot = null;
			if (pagemgr == null) 
				{
				scratchroot = new File(docroot + File.separator + 
								"WEB-INF" + File.separator + "molly_tmp");

			//mkdirs for /a/b/c.java makes THREE dirs: 'a', 'b' and 'c.java'
			//but that's not an issue here.
				if (! scratchroot.exists())
					scratchroot.mkdirs();
					
				pagemgr = new PageMgr(this, docroot, scratchroot, log);
				}		

			error_page 	= WebUtil.getParam(this, "error_page", null);
			page_404 	= WebUtil.getParam(this, "404_page", null);
	
			log.info("PageServlet init() finished");
			log.info("Document root=", docroot);
			log.info("Scratch root=", scratchroot);
			log.info("Error page=", error_page);
			log.info("404   page=", page_404);
			}
		}	
	catch (Exception e) {
		throw new ServletException(e);
		}
	}

protected void service(HttpServletRequest req, HttpServletResponse res)
throws ServletException, java.io.IOException
	{
	//The servlet engine fills in the servletpath appropriately for
	//extension mapped servlets, so for example, if we have mapped
	//*.mp or *.jsp to PageServlet, then when PageServlet gets the
	//request for /a/b/c.mp, the servletpath in the request object
	//will be /a/b/c.mp
	
	//tomcat(piece of shit) sends context=/foo, servletpath=bar.jsp
	//for /foo/bar.jsp even for the *DEFAULT* (/) context.
	//So we say:
	//   String path = req.getContextPath() + req.getServletPath();
	//looks like tomcat(piece of shit) may or may not have fixed the
	//above bug. i just don't give a shit about tomcat anymore.
	//
	//The page path is relative from the context. The pagemgr was created
	//with the appropriate docroot that includes the webapp context (if any). 
	//We just need to pass it the servlet path to get the page.
	String path = req.getServletPath();

	if (dbg) log.bug("Enter request for path: ", path);

	if (path == null)
		throw new ServletException("Don't know how to handle this request, servletpath was null");

	try {
		if (dbg) {
			System.out.format("PageServlet: INCLUDE info: uri=%s, c_path=%s, s_path=%s, path_info=%s, q=%s", 
			(String) req.getAttribute("javax.servlet.include.req_uri"),
			(String) req.getAttribute("javax.servlet.include.context_path"), 
			(String) req.getAttribute("javax.servlet.include.servlet_path"), 
			(String) req.getAttribute("javax.servlet.include.path_info"),
			(String) req.getAttribute("javax.servlet.include.query_string"));
			
			System.out.println();
			
			System.out.format("PageServlet: FORWARD info: uri=%s, c_path=%s, s_path=%s, path_info=%s, q=%s", 
			(String) req.getAttribute("javax.servlet.forward.req_uri"),
			(String) req.getAttribute("javax.servlet.forward.context_path"), 
			(String) req.getAttribute("javax.servlet.forward.servlet_path"), 
			(String) req.getAttribute("javax.servlet.forward.path_info"),
			(String) req.getAttribute("javax.servlet.forward.query_string"));

			System.out.println();
			}
	
		final String include_path =
			(String) req.getAttribute("javax.servlet.include.servlet_path");

		if (include_path != null) {	
			//RequestDispatcher include ?
			path = include_path;
			}
		else{
		/*
			WE DONT DO THIS SINCE forward.* properties are for the ORIGINAL
			REQUEST. the path is already properly set for forwards.
			
			final String forward_path =
				(String) req.getAttribute("javax.servlet.forward.servlet_path");
			//RequestDispatcher forward ?
			if (forward_path != null) {
				path = forward_path;
				}
		*/
		}
		
		if (dbg) log.bug("Asking PageMgr for the page: ", path);
		Page page = pagemgr.getPage(path); 
		if (dbg) log.bug("PageMgr returned page: [", path, "]:", page);
		
		if (page == null) 
			{
			if (page_404 == null) {
				throw new ServletException("Molly page:" + path + " not found.");
				}
			else{
				WebUtil.clientRedirect(req, res, page_404);
				return;
				}
			}
			
		page.render(req, res);
		}
	catch (Exception e)
		{
		//this helps us figure out bugs in the "error.mp" page itself (which
		//otherwise hard to see, since error.mp -> parse error --> error.mp)
		if (dbg) log.bug(IOUtil.throwableToString(e)); 
		
		//error occured after page partially shown to user
		if (res.isCommitted() && error_page != null) 
			{	
			req.setAttribute("javax.servlet.error.status_code", 500);
			req.setAttribute("javax.servlet.error.exception", e);
			req.setAttribute("javax.servlet.error.request_uri", path);
			req.setAttribute("javax.servlet.error.servlet_name", "Molly Page Servlet");

			RequestDispatcher rd = req.getRequestDispatcher(error_page);
			rd.include(req, res);
			return;
			}
		
		//page hasn't been shown to the user. The container will set all
		//the javax.servlet.error.* properties and forward to the error
		//page (if any) that is configured for the container.
		ServletException se = new ServletException(e.toString());
		se.setStackTrace(e.getStackTrace());
		throw se;
		}
	}
            
public String toString() {
	return  "Scrumptious and delicious Page Servlet ! (See: www.mollypages.org)";
	}	
	
public void destroy() 
	{
	try {
		pagemgr.destroy();
		pagemgr = null;
		super.destroy();
		}
	catch (Exception e) {
		System.out.println(IOUtil.throwableToString(e));
		}
	}	
	
}
