// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.servlet;

import java.io.*;
import java.util.logging.*;
import java.util.*;
import javax.servlet.*;
import javax.servlet.http.*;

import fc.io.*;
import fc.web.*;
import fc.util.*;

//In the future, java.security should allow us to use logon credentials for
//security. Right now, as of jdk 1.4 it seems to be non-trivial to create
//and install new security policies based on the current login info. java.security
//is really targetted towards code principals i.e, granting *code* various permissions
//based on where the code was *loaded* from. 

/** 
Manages various {@link Action} available to a servlet. Subclasses 
should add and manage appropriate actions.

@author	hursh jain
**/
public abstract class ActionMgr
{
protected	final	SystemLog	log; 
protected 	final	Map			actionMap;

/** 
Constructs a new ActionMgr 
**/
protected ActionMgr() 
	{
	log = Log.getDefault();
	actionMap = new HashMap();
	}

/** 
Registers an action. Action names are case <u>insensitive</u>.
**/
public final void addAction(Action action) 
	{
	//log.entering("ActionMgr", "addAction"); 
	
	String name = action.getName().toLowerCase();
	
	if (actionMap.containsKey(name)) {
		throw new IllegalArgumentException("An action with the name '" + name + " already exists"); 
		}
	actionMap.put(name, action);
	
	log.bug("ActionMgr: installed new action=", name);
	}

/** 
Handles a request from a web client. Expects certain parameters to
be present in the specified <tt>HttpServletRequest</tt>. The main
parameter needed by this method is:
<blockquote>
<ul>
	<li>Param name:<tt>act</tt><br>
	Param value: A case-insenstive class name of the {@link Action}
    class that will be invoked (for example: "showdir"). Case is not
    important but using all lower case names will save a few machine 
	cycles. This should match the name with which a particular action
	object was constructed.
</ul>
</blockquote>
Various actions expect action-specific parameters via the request
object parameter. This parameters can be set programatically or via
the web client and optionally, actions can also return results via 
setting them in the request object. The documentation for each action
method should give more detail.

@throws	Exception if the action parameter is missing or not understood
**/
public void handleWebRequest(HttpServletRequest req, HttpServletResponse res) 
throws Exception
	{
	log.bug("enter");
	if (log.canLog(SystemLog.DEBUG)) {
		log.bug("requested URL=" + WebUtil.getRequestURL(req), "req=", req);
		}
	String actionName = WebUtil.getRequiredParam(req, "act").toLowerCase();
		
	Action action = null;
	if (actionMap.containsKey(actionName)) {
		action = (Action) actionMap.get(actionName);
		}
	else { //action not found
		log.warn("Could not find action for action=" + action, "installed actions=", actionMap);
		throw new ServletException("ActionMgr: Could not understand the web request, request=" + req);	
		}
	
	action.handle(req, res);	//the op will extract further params as needed
	log.bug("exit");
	}
	
public String toString() {
	return getClass().getName() + "; actions=" + actionMap;
	}

}          //~class ActionMgr
