// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.servlet;

import java.io.*;
import java.net.*;
import javax.servlet.*;
import javax.servlet.http.*;

import java.util.*;

import fc.io.*;
import fc.web.*;
import fc.util.*;

/** 
An abstract servlet filter that only allows authenticated access to a
resource. This filter will redirect the user to the <tt>login_page</tt> if
the user is not logged in. The originally requested target URL (if any)
will be saved via the method. The saved URL will be complete, i.e., will
start from the protocol string <tt>http..</tt> upto and including any query
string. Note: if the original URL was submitted via POST, the POST data is
<u>not</u> saved, only the target URL is saved.
<p>
There are 2 ways that the original URL can be saved. One is to save it as a
temporary cookie. This implies that the client must have cookies enabled.
The other way is to tack it on to the URL as a url parameter. <u>In either
case, the original target is saved as a parameter with
name=<tt>login.target</tt> and value=<tt>URLEncoder.encode(target_page)</tt></u>
<p>
The following keys must be specified as the init parameters.
<ul>
	<li>key name: <tt>appName</tt>, the name of the webapp (the name is an
	arbitrary string but must be the same as is specified in the init parameter
	of {@link WebApp}).
	<li>key name: <tt>login_page</tt>, the value should be an absolute
	path (from the website's document root) to the login page or URL.
The following key is optional:
	<li>key name: <tt>use_cookie</tt>, the value should be <tt>"true"</tt>
	or <tt>"false"</tt>. <tt>true</tt> means save the original target page
	as a cookie, <tt>false</tt> means save that page as part of the URL.
	Defaults to <tt>false</tt>
</ul>

<b>Note:</b> this class cannot be directly instantiated (since it's
abstract) and should not be specified as the name of some filter in the
servlet container's web.xml file.

@author hursh jain
**/
public abstract class AuthFilter implements Filter 
{
private static 	final boolean dbg = false;

boolean			use_cookie = true;
FilterConfig 	config;
ServletContext 	context;
String 			login_page;
String 			appName;
  
public void init(FilterConfig config) throws ServletException 
	{
	this.config = config;
	this.context = config.getServletContext();
	login_page = config.getInitParameter("login_page");

	if (login_page == null)
		throw new ServletException("AuthFilter: the login_page parameter was missing");
	
	appName = config.getInitParameter("appName");
	if (appName == null)
		throw new ServletException("AuthFilter: the appName parameter was missing");
		
	String tmp = config.getInitParameter("use_cookie");
	if (tmp != null)
		this.use_cookie = Boolean.valueOf(tmp).booleanValue();
	}

public void destroy() 
	{
	config = null;
	context = null;
	}

public void doFilter (
	final ServletRequest req, final ServletResponse res, final FilterChain chain) 
throws ServletException, IOException
	{
	if (dbg) System.out.println(">>>AuthFilter: doFilter() START");
	
    if ( (! (req instanceof HttpServletRequest)) ||
		 (! (res instanceof HttpServletResponse)) ) 
		{
      	throw new ServletException("Request type was not HTTP/HTTPS");
	  	}
		
    final HttpServletRequest  request = (HttpServletRequest) req;
	final HttpServletResponse response = (HttpServletResponse) res;

	boolean loggedin = false;
	try {
		loggedin = isUserLoggedIn(request, response);
		}
	catch (Exception e) {
		throw new ServletException(e);
		}
		
	if (! loggedin ) 
		{
		if (dbg) System.out.println(">>>AuthFilter: user not logged in");
		showLoginPage(request, response);
		}
	else {
		if (dbg) System.out.println(">>>AuthFilter: user logged in"); 
		//only if user is logged in
		chain.doFilter(req, res);
		}

  	if (dbg) System.out.println(">>>AuthFilter: doFilter() END");
	}
	
/**
Redirects the user to a login page. The login page must set login status in
some way understandable by invokations of this method.
*/
protected void showLoginPage(
	final HttpServletRequest req, final HttpServletResponse res)
throws ServletException, IOException
	{
	/*
	 requestURL gives us the protocol, port, the entire URL
	 (including path info) whereas requestURI would only give
	 us the entire path but not the protocol, port etc. We
	 save the full URL and the query string
	*/
	final StringBuffer buf = req.getRequestURL();

	final String query = req.getQueryString();
	if (query != null) {
		buf.append('?');
		buf.append(query);
		}

	//we encode because the login target may itself contain
	//chars like ? and = etc
	String login_target = URLEncoder.encode(buf.toString());

	//set target to original request page
	if (use_cookie) {
		final Cookie cookie = new Cookie("target", login_target);
		cookie.setPath("/");
		cookie.setMaxAge(-1); //only need it for browser session
		res.addCookie(cookie);
		if (dbg) System.out.println("setting cookie: " + cookie.getValue());
		}
	else {
		login_page =  login_page + "?target=" +  login_target;
		}
	
	res.sendRedirect(login_page);  //login_page NOT target	
	}

/**
This method should somehow check to see if the user is logged in or not.
Typically, this will be done via getting a session_id (either from a cookie
or a URL) and using that session_id to search for a memory or database
session data to see if that session still exists and has not expired.
*/
public abstract boolean isUserLoggedIn(
	HttpServletRequest req, HttpServletResponse res)
throws Exception;
			
} //~class AuthFilter

