// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.servlet;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.*;
import fc.web.servlet.*;
import fc.io.*;

/**
Sends the specified file as a binary stream after setting the
ContenEncoding to be gzip. (the specified file should have
a .gz extension, this is mandated because it helps catch
user errors, for example forgetting to gzip the file).
<p>
<b>Note, this servlet does not Gzip data, it only serves data that
has already been gzip'ed.</b>
<p>
To use, add something like the following to web.xml:
<blockquote>
<pre>
&lt;servlet&gt;
	&lt;servlet-name&gt;gzip&lt;/servlet-name&gt;
	&lt;servlet-class&gt;fc.web.servlet.GzipFileServlet&lt;/servlet-class&gt;
&lt;/servlet&gt;
...
&lt;!-- an example mapping, your taste may vary --&gt;
&lt;servlet-mapping&gt;
  &lt;servlet-name&gt;gzip&lt;/servlet-name&gt;
  &lt;url-pattern&gt;/gzip&lt;/url-pattern&gt;
&lt;/servlet-mapping&gt;
</pre>
</blockquote>
Then, in this example, one can say, in your HTML document:
<blockquote>
<pre>
&lt;html&gt;
&lt;head&gt;
	&lt;title&gt;Untitled&lt;/title&gt;
	<font color=blue>&lt;script src="/gzip?js=myscript.js"&gt;&lt;/script&gt;</font>
&lt;/head&gt;
&lt;body&gt;
hello
&lt;/body&gt;
&lt;/html&gt;
</pre>
</blockquote>
That will load the <i><code>myscript.js.gz</code></i> 
file (and if myscript.js.gz is not present, a servlet exception
will be logged). Note, the ".gz" extension is added automatically
if not specified in the filename.
<p>
There are two modes in serving files. In both cases, the file
to be retrieved <u>must already exist as a gzip file on 
the server</u>.
<ol>
<li>
Retrieve a arbitrary gzip'ed file from the server. There are 3 
required parameters:
<dl>
<dt>file</dt>
<dd>the filename (if the specified name does not have a 
.gz extension, ".gz" will be added automatically to the
specified name). [see <a href="#path"> path note below</a>]
</dd>
<dt>mimetype</dt>
<dd>The mimetype of the retrieved file (<i>after</i> it has been
gunzipped). This will commonly be <code>text/html</code> or 
<code>text/javascript</code></dd>
<dt>encoding</dt>
<dd>The text encoding of the specified file. Commonly, it is
US-ASCII, ISO-8859-1 or UTF-8</dd>
</dl>
Example: <code>
/gzip?file=foo.html.gz&mimetype=text/html&encoding=ISO-8859-1
</code>
</li>

<li>
Retrieve a Javascript file (convenience method): This will automatically
set the  content-type to be <i>text/javascript; charset=utf-8</i>
(so the mime or charset do not have to be specified). One need
only specify:
<dd>
<dt>js</dt>
<dl>path to the gzip'ed file. [see <a href="#path"> path note below</a>]</dl>
</dd>
</li>
</ol>
<hr>
<h3>Path name for the specified file</h3>

This path to the included file must be in absolute (start with a
'/') or relative form.
<p>
Absolute path names are like HTML absolute names and 
start from the document root directory of the web server. 
<p>
Relative names (relative to the invoking page) are <b>not</b>
supported by this servlet.

@author hursh jain
*/
public final class GzipFileServlet extends javax.servlet.http.HttpServlet
{
/* 
  context        page  
  /WEBAPP/      /bar/baz/foo.js
  /WEBAPP/      /bar/img/zap.img
  
  from foo.js -> 
  	a)  "../img/zap.img"  [relative]
	b)	"/WEBAPP/bar/img/zap.img" [absolute #1]
	c)	"/bar/img/zap.img"     [absolute #2, WEBAPP added by us]
   
  we really want (a) and (b), (c) is not-intuitive.
  (a) and (b) correspond to tradional relative, absolute paths
  (where absolute is always from the web server document root]

 (a) is too tricky, we don't know the path to the invoking
     page in this servlet (so cannot do relative to that 
     unknown path).
*/

File docroot;

String d_charset = "text/html; charset=ISO-8859-1";

public void init(ServletConfig conf) throws ServletException 
	{
	super.init();
	ServletContext context = conf.getServletContext();
	ServletContext root_context = context.getContext("/");
	docroot = new File(root_context.getRealPath("/"));
	}
	
/**
Returns the specified gzip file. If the specified name 
does not end with ".gz", ".gz" will automatically
be tacked on to the specified name (so files served by
this method must always end with ".gz"
*/	
public void doGet(final HttpServletRequest req, final HttpServletResponse res) 
throws ServletException, IOException
	{
	final String file = req.getParameter("file");
	if (file != null) {
		sendFile(file, req, res);
		return;
		}
	
	final String js = req.getParameter("js");
	if (js != null) {
		sendJSFile(js, req, res);
		return;
		}
	
	throw new ServletException("Bad request. I need either the \"file\" or the \"jsfile\" parameter.");
	}
	
void sendFile(final String file, final HttpServletRequest req, final HttpServletResponse res) 
throws ServletException, IOException
	{
	final String mime	   = WebUtil.getRequiredParam(req, "mimetype");
	final String encoding  = WebUtil.getRequiredParam(req, "encoding");
	final String content_type = mime + "; charset=" + encoding;
	res.setContentType(content_type);
	res.setHeader("Content-Encoding", "gzip");
	ServletOutputStream out = res.getOutputStream();
	push(out, file);
	}

String js_content_type = "text/javascript; charset=utf-8";

void sendJSFile(final String file, final HttpServletRequest req, final HttpServletResponse res) 
throws ServletException, IOException
	{
	res.setContentType(js_content_type);
	res.setHeader("Content-Encoding", "gzip");
	final ServletOutputStream out = res.getOutputStream();
	push(out, file);
	}

void push(final ServletOutputStream out, final String file) 
throws IOException
	{
	final File f = makeFile(file);
	final byte[] buf = IOUtil.fileToByteArray(f);
	final BufferedOutputStream bout = new BufferedOutputStream(out);
	bout.write(buf, 0, buf.length);
	bout.flush();
	bout.close();
	}

File makeFile(String path) throws IOException
	{
	if (! path.endsWith(".gz")) {
		path += ".gz";
		}
		
	File f = new File(docroot, path);

	if (! f.exists() || ! f.isFile()) {
		throw new IOException("\nERROR: file: [" + path + "] not found");
		}
	return f;
	}
}
