// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.servlet;

import java.io.*;
import java.net.*;
import java.sql.*;
import java.util.*;
import java.security.*;
import javax.servlet.*;
import javax.servlet.http.*;

import fc.jdbc.*;
import fc.io.*;
import fc.web.*;
import fc.util.*;

/** 
Logs in/logs out the user. Works in conjunction with {@link JDBCAuthFilter} and
the login page of the application. (handles the submit from the HTML login
form). Moreover, the application can logout the user by invoking this servlet
with an additional <tt><b>act=logout</b></tt> query string. Uses {@link
JDBCSession} to create/delete session id's automatically on successful
logins/logouts.
<p>
Requires the following servlet initialization parameters
<ul>
	<li><tt>welcome_page</tt> the <i>webapp relative</i> path to the
	welcome page to be shown after a successful login.
	<li><tt>login_page</tt> the <i>webapp relative</i> path to the
	login page.
	<li><tt>logout_welcome_page</tt> the <i>webapp relative</i> path
	to the welcome page to be shown after a successful logout. (this
	parameter is <b>optional</b> and if not specified, the welcome_page
	will be used).
</ul>
<u>On login failure</u>, the following attributes are set in the request
before control is transferred to the login page via a server side
redirect.
<ol>
	<li><tt>retrycount</tt>, value is a Integer object representing the
	number of times login has been unsuccessfuly tried. Note: <u>the login
	page should read this attribute if present and store it in the form
	as a hidden parameter. When the login form is submitted, this variable
	will be sent back to this servlet in the request as a parameter and
	upon login failure, be appropriately incremented.</u>
</ol>
<u>On login success</u>, the following attributes are set as a cookie on
the client. This cookie is removed on logout.
<ol>
	<li><tt>{@link #SID_COOKIE_NAME}</tt> the session ID assigned to the
	user. After logging in, a session will exist in the database. {@link
	JDBCSession} can thereafter be used to store any information for that
	session in the database via the session ID.
	<li><tt>user.name</tt>, the name of the user that was succefully
	used to login to the system. (this is useful for displaying the
	username in the front end page without hitting the database everytime).
</ol>
In addition, upon successful login, the {@link #onLogin} method is invoked.
This method can be overriden as necessary by subclasses. Similarly, In addition, upon 
successful logout, the {@link #onLogout} method is invoked. 
<p>
The servlet requires the following request parameters from the
login form:
<ul>
	<li><tt>username</tt>
	<li><tt>password</tt>
</ul>
The following request parameters are optional. 
<ul>
	<li><tt>target</tt> if present (either as a cookie or in the URL),
	the client is redirected to the URL specified by the target
	(otherwise the client is redirected to the welcome_page after
	login/logout). The {@link AuthFilter} automatically stores the
	original target page as a parameter (URLEncoded) so that users are
	seamlessly redirected to their original target after a successful
	login or logout.
</ul>
<p>
Requires the following database schema:
<blockquote>
A <b>Users</b> table must exist. (note: "user" is a reserved word in many
databases, the table must be called User<b>s</b>.
<ol>
The following columns must exist in the <tt>Users</tt> table.
	<li> <b>user_id</b> the user id
	<li> <b>username</b> the name of the user (corresponds to
	the username parameter in the login form)
	<li> <b>password</b> the password for the user (corresponds
	to the password parameter in the login form).
</ol>
The class will use the {@link fc.jdbc.dbo.DBOMgr} framework so a
<b>UserMgr</b> class corresponding to the aforementioned <b>User</b> table
must exist in the classpath of this servlet.
</blockquote>
Since this class uses {@link JDBCSession}, the default database
tables required by {@link JDBCSession} also must exist.
<p>
For security reasons, for logging in, the username/password form must be
submitted via a POST (GET is fine when logging out).

@author hursh jain
**/
public class LoginServlet extends FCBaseServlet
{
/** value = "sid" **/
public static final String SID_COOKIE_NAME = "sid";

protected String  		login_query;
protected String 		loginPage;
protected String 		welcomePage;
protected String 		logoutWelcomePage;
protected JDBCSession	session;
protected MessageDigest hash;

public void init(ServletConfig conf) throws ServletException
	{
	super.init(conf);

	try {
		loginPage 			= WebUtil.getRequiredParam(this,  "login_page");
		welcomePage 		= WebUtil.getRequiredParam(this,"welcome_page");
		logoutWelcomePage 	= WebUtil.getParam(
								  this, "logout_welcome_page", welcomePage);
		login_query 		= WebUtil.getRequiredParam(this, "login_query");
		String hash_name	= WebUtil.getParam(this, "password_hash", null);
		
		log.bug("login page=", loginPage);
		log.bug("welcome page=", welcomePage);
		log.bug("logout welcome page=", logoutWelcomePage);
		log.bug("login_query=", login_query);
		log.bug("password_hash=", hash_name);

		hash 	= (MessageDigest) MessageDigest.getInstance(hash_name);
		session	= JDBCSession.getInstance();
		}
	catch (Exception e) {
		log.error(IOUtil.throwableToString(e));
		throw new ServletException(IOUtil.throwableToString(e));
		}
	}

/**
Returns the cookie corresponding to the "sid". (this cookie
has key = {@link #SID_COOKIE_NAME} and the value is the SID
created/set at login time). Returns <tt>null</tt> is no
sid cookie is found.
*/
public static Cookie getSIDCookie(HttpServletRequest req)
	{
	Cookie[] cookies = req.getCookies();
	
	if (cookies == null)
		return null;
		
	for (int n = 0; n < cookies.length; n++) 
		{
		if (cookies[n].getName().equals(SID_COOKIE_NAME))
			{
			return cookies[n];
			}
		}
	return null;
	}

boolean checkHasValidSID(Connection con, HttpServletRequest req) 
throws SQLException
	{
	Cookie c = getSIDCookie(req);
	if (c == null)
		return false;
		
	String sid = c.getValue();
	if (session.exists(con, sid))
		return true;
		
	return false;
	}

public void doGet(HttpServletRequest req, HttpServletResponse res)
throws ServletException, IOException
	{
	//to support logout only.
	doPost(req, res);
	}
	
public void doPost(HttpServletRequest req, HttpServletResponse res)
throws ServletException, IOException
	{
	Connection con = null;
	
	try {
		con = getConnection();
		
		String action = req.getParameter("act");

		if (action != null) {
			if (action.equals("logout")) 
				{
				if (doLogout(con, req, res)) {
					goToTargetPage(req, res, logoutWelcomePage);
					return;
					}
				return;
				}
			}
			
		//at this point we are logging in.
		
		//may have valid SID if login page accessed directly
		//via browser's history or whatever.
		if (checkHasValidSID(con, req)) {
			goToTargetPage(req, res, welcomePage);
			return;
			}

		if (! doLogin(con, req, res)) {
			addFailureMessage(req);
			WebUtil.forward(req, res, loginPage);//server side redirect
			return;
			}
		else{
			goToTargetPage(req, res, welcomePage);
			return;
			}
		}
	catch (Exception e) {
		throw new ServletException(e);
		}
	finally {
		try {
			con.close();
			}
		catch (SQLException e) {
			throw new ServletException(e);
			}
		}
		
	} //~doPost

/**
This method validates the specified username/password. 
<p>
<b>
This class should be subclassed to override this method and validate the
supplied username/password against a database in a different fashion if
desired.
</b>
The default implmentation of this method works with the following
initialization parameters.
<ul>
<li><tt>login_query</tt> (required): the query string to validate if this
username/passwrod combination exists. This query string should in
<b>PreparedStatement</b> format with 2 question marks (the first one will
be set with the username and the 2nd with the password). </li>
<li><tt>password_hash</tt> (optional): if present, should contain the 
name of the java cryto hash function to hash the password before comparing
it with the database (this is for cases where the passwords are stored as
hashed values in the database). Examples include <tt>MD5</tt>, <tt>SHA-1</tt>
etc.
</li>
</ul>
This method should return the following values:
<ul>
<li>If authentication failed: <tt>null</tt></li>
<li>If authentication succeeded: 
	<ol>
	<li>If the returned string is non-null and non-empty, then it should
	contain the userid for the authenticated user and this
	<tt>userid</tt> is stored in the {@link JDBCSession}.</li>
	<li>If the user authenticated succeeds, a non-null string containing
	a unique username or userid should be returned. (the username/userid
	should be the Primary key field used in the database to uniquely 
	identify a user).
	</li>	
	</ol>
</li>
</ul>
*/
public String validateUser( 
  Connection con, String username, String password) 
throws SQLException, IOException
	{
	PreparedStatement pstmt = null;

	if (! (con instanceof fc.jdbc.PooledConnection) ) { 
		pstmt = con.prepareStatement(login_query);
		}
	else{
		PooledConnection pc = (PooledConnection) con;
		pstmt = pc.getCachedPreparedStatement(login_query);
		}
		
	pstmt.setString(1, username);
	pstmt.setString(2, encodePassword(password));

	log.bug("Validating logon using: ", pstmt);
		
	ResultSet rs = pstmt.executeQuery();
	String uid = null;
	
	if (rs.next()) {
		log.bug("Successfully validated username=", username, ". Got uid=[", uid + "]");
		uid = rs.getString(1);
		}
		
	return uid; 
	}

/**
For this method to be available from application code, this servlet
should be set to load on startup and code similar to the following
example invoked.
<blockquote>
<tt>
LoginServlet ls = (LoginServlet) WebApp.allServletsMap.get("fc.web.servlet.LoginServlet");
if (ls == null) { //can happen if servlet is not loaded yet
	throw new Exception("Unexpected error: LoginServlet was null");
	}
return ls.encodePassword(passwd);
</tt>
</blockquote>
*/
public String encodePassword(String password) throws IOException
	{
	if (hash == null || password == null)
		return password;

	try {
		//MessageDigest getInstance not thread safe, alternative is to
		//use that in a sychronized block.
		final MessageDigest hash2 = (MessageDigest) hash.clone();
		final byte[] digest_buf = hash2.digest (password.getBytes("UTF-8"));
		
		//postgres, mysql have md5() which returns the md5 hash as a 
		//hexadecimal string. they don't really have base64 that can work
		//seamlessly as:  string->md5()->base64-->password		
		//   password = new sun.misc.BASE64Encoder().encode(digest_buf);
		//So we do this:
		
		password = HexOutputStream.toHex(digest_buf);
		}
	catch (Exception e) {
		throw new IOException(IOUtil.throwableToString(e));
		}
	
	return password;
	}

/**
Returns true on a successful login, false otherwise.
*/
boolean doLogin(
 Connection con, HttpServletRequest req, HttpServletResponse res) 
throws SQLException, IOException
	{
	String username = req.getParameter("username");
	String password = req.getParameter("password");	

	if (username == null || password == null) {
		log.error("username not recieved in request (not present at all)");
		return false;
		}

	if (password == null) {
		log.error("password not recieved in request (not present at all)");
		return false;
		}

	String sid = SessionUtil.newSessionID();
	String uid = validateUser(con, username, password);
	boolean suxez = (uid != null);
	
	if (suxez) 
		{
		session.create(con, sid, uid);
			
		addCookie(res, SID_COOKIE_NAME, sid);
		addCookie(res, "user.name", username);
		onLogin(con, sid, username, req, res);
		}
	else{
		log.info("login FAILED for username=[", username, "]");
		}
		
	return suxez;
	}

boolean doLogout(Connection con, 
				 HttpServletRequest req,  HttpServletResponse res) 
throws SQLException, IOException
	{
	Cookie c = getSIDCookie(req);

	if (c == null)
		return false;

	String sid = c.getValue();
	log.bug("logging out: ", sid);

	boolean suxez = false;	
	if (session.exists(con, sid)) {
		session.expire(con, sid);
		suxez = true;
		}
	else{
		log.bug("session:", sid, " does not exist, ignoring logout");
		}
		
	deleteCookie(res, "user.name");
	deleteCookie(res, SID_COOKIE_NAME);
	onLogout(con, sid, req, res);
	
	return suxez;
	}

/**
This method is invoked upon successful login. By default, it does
nothing but subclasses can override this method as needed.

@param	con			a connection to the database
@param	username	the username for this user (that was used to login the
					user via the login query)
@param	sid			the session id for this user
*/
public void onLogin(Connection con, String sid, String username,
				 HttpServletRequest req,  HttpServletResponse res) 
throws SQLException, IOException
	{
	/* override as necessary */
	}

/**
This method is invoked upon successful login. By default, it does
nothing but subclasses can override this method as needed.

@param	con			a connection to the database
@param	sid			the session id for this user
*/
public void onLogout(Connection con, String sid,
				 HttpServletRequest req,  HttpServletResponse res) 
throws SQLException, IOException
	{
	/* override as necessary */
	}

void goToTargetPage(
	HttpServletRequest req, HttpServletResponse res, String defaultPage)
throws IOException, ServletException
	{
	String target = null;
	
	final Cookie cookie = WebUtil.getCookie(req, "target");
	if (cookie != null)	{
		target = cookie.getValue();
		cookie.setPath("/");
		cookie.setMaxAge(0); //delete
		res.addCookie(cookie);
		}
	else	
		target = req.getParameter("target");

	if (target == null) 
		{
		//we want browser's url to change so client side
		WebUtil.clientRedirect(req, res, defaultPage);
		}
	else {
		/*
		we don't use a server side redirect because the target
		is the full original URL (like: http://....) and
		server side redirect would try /http://.... (the
		leading / because server side redirects are context
		relative) we also want the browser's url to change
		*/
		target = URLDecoder.decode(target);
		
		WebUtil.clientRedirect(req, res, target);
		}
	}

void addFailureMessage(HttpServletRequest req)
throws ServletException	
	{
	String retrycount = req.getParameter("retrycount");
	if (retrycount == null)
		retrycount = "0";
	
	Integer rint = null;
	try { 
		rint = new Integer(Integer.parseInt(retrycount) + 1);
		}
	catch (NumberFormatException e) {
		rint = new Integer(1);
		log.warn(e);
		}

	req.setAttribute("retrycount", rint);
	}	
	
final void addCookie(HttpServletResponse res, String key, String val) 
	{
	Cookie cookie = new Cookie(key, val);
	cookie.setMaxAge(session.getExpireTime()); 
	res.addCookie(cookie);
	}

final void deleteCookie( HttpServletResponse res, String key) 
	{
 	//the value is irrelevant since we are deleting the cookie
	Cookie cookie = new Cookie(key, ""); 
	cookie.setMaxAge(0); 
	res.addCookie(cookie);
	}

}  //~class

