// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.servlet;

import javax.servlet.http.*;
import java.security.*;
import java.sql.*;
import java.io.*;

import fc.jdbc.*;
import fc.io.*;
import fc.web.*;
import fc.util.*;
 
/**  
Modified version of CommandToken class from the book: <blockquote><tt>
Web dev. with JSP by fields, kolb and bayern </tt></blockquote> Uses
{@link JDBCSession} to store data.
<p>
Essentially, a token prevents repeating a page action upon back-button, reload,
etc of a page. (for example, reprocessing an order if an order page was
reloaded). This is done by setting a transaction token as a hidden field in the
page we want to protect and also setting the same token in the user session.
(This setting is done by other pages/servlets that send the use to the
protected page, for example, from an earlier html page). When the protected
order page is submitted, the order processing code checks to see if the
session-token and the submitted form-token match. If so, the order is run
<b>and</b> the session-token deleted. 
<p>
The next time the protected page is reloaded and submitted, the session-token
is missing on the server, hence the submitted form (which still has the earlier
token) and session token will not match and hence the order is not rerun.
<p>
Thread safety: Methods in this class are not thread safe and should be
called via higher level synchronization (typically on the session object
for a given user);

@author hursh jain
**/ 
public class TransactionToken
{
/** The token is stored in the session with this key **/
public static final String  TransactionToken_Key = "transact_token";

static Log 			log 	= Log.get("fc.web.servlet");
static JDBCSession session = JDBCSession.getInstance(log);

/** 
Creates a new transactional token. Tokens are unique per
session. Calling this method more than once will replace a
prior token in (if any) in the session. 

@param	con			connection to database used by {@link JDBCSession}
@param	sessionID	the sessionID of the client

@throws IllegalStateException	if the specified sessionID is expired/not valid
**/
public static void create(Connection con, String sessionID) 
throws SQLException
	{
 	if (! session.exists(con, sessionID))
 		throw new IllegalStateException("the specifed sessionID [" + sessionID + "] does not exist");

 	long currentTime = System.currentTimeMillis();
    byte[] time  = String.valueOf(currentTime).getBytes();
    byte[] id = sessionID.getBytes();
	MessageDigest md5 = null;     
	String token = null;
	try { 
		md5 = MessageDigest.getInstance("MD5");
		md5.update(id);
    	md5.update(time);
		token = toHex(md5.digest()).toString();
   		}
    catch (Exception e) {
  		throw new IllegalStateException(IOUtil.throwableToString(e));	
   		}
	
	log.bug("Adding new transaction token=", token, " for sessionID=", sessionID); 
	session.add(con, sessionID, TransactionToken_Key, token);
  	}

/**
Revokes the transactionID (if any) from the session
data

@param	session		the JDBC session to save the token to
@param	sessionID	the sessionID of the client
**/
public static void revoke(Connection con, String sessionID) 
throws SQLException
	{
	session.delete(con, sessionID, TransactionToken_Key);
	}

/** 
Searches for a token in the request (under the parameter name
{@link #TransactionToken_Key} and tries to match it with a
corresponding token in the session. Returns <tt>true</tt> if the
tokens match (and hence the token is valid), <tt>false</tt>
otherwise.
**/
public static boolean isValid(
	Connection con, String sessionID, HttpServletRequest req) 
throws SQLException
	{
    String requestToken = req.getParameter(TransactionToken_Key);
    if (requestToken == null) {
		return false;
      	}
    else{		    
    	String sessionToken = session.get(
    							con, sessionID, TransactionToken_Key);
      	//sessionToken can be null (if not set or session expired)
      	//or some bad value
      	return requestToken.equals(sessionToken);
  		}
  }

private static StringBuffer toHex(byte[] digest) 
 	{
    StringBuffer buf = new StringBuffer();
    for (int i=0; i < digest.length; i++)
	      buf.append(Integer.toHexString((int)digest[i] & 0x00ff));
    return buf;
  	}

public static void main(String args[]) throws Exception
	{
	Args myargs = new Args(args);
	myargs.setUsage("java fc.web.servlet.TransactionToken -conf conf-file -sessionID sessionID");
	
	String propfile = myargs.getRequired("conf");
	String sessionID = myargs.getRequired("sessionID");
	
	ConnectionMgr mgr = new SimpleConnectionMgr(  
							new FilePropertyMgr(
								new File(propfile)));
	Connection con = mgr.getConnection();
	
	log.setLevel(Log.DEBUG);

	JDBCSession session = JDBCSession.getInstance(log);
	session.create(con, sessionID);
	
	System.out.println("creating new transaction id...");
	TransactionToken.create(con, sessionID);
	System.out.println("Getting new transaction id from database");
	String tok = session.get(con, sessionID, TransactionToken_Key);
	System.out.println("transaction id = " + tok);
	System.out.println("revoking token..");
	TransactionToken.revoke(con, sessionID);
	System.out.println("....revoked");
	session.get(con, sessionID, TransactionToken_Key);
	System.out.println("transaction id = " + tok);

	con.close();
  	}
}
