// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.servlet;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.*;
import java.sql.*;

import fc.io.*;
import fc.jdbc.*;
import fc.util.*;
import fc.util.cache.*;
import fc.web.*;
import fc.web.forms.*;

/** 
Application level global object running within a servlet web application
(i.e., a servlet context). Global webapp data can be stored/retrieved
via the put/get methods.
<p>
Initializes and stores various variables useful for all servlets/pages
running in our JVM. Implements {@link javax.servlet.ServletContextListener}
and initializes itself when informed by the servlet container's context initialization event.
<p>
It's optional to use this class. If it is used, it's configured by adding the
following to the appropriate sections of WEB-INF/web.xml:
<blockquote>
<pre>
&lt;context-param&gt;
        &lt;param-name&gt;configfile&lt;/param-name&gt;
        &lt;param-value&gt;app.conf&lt;/param-value&gt;
&lt;/context-param&gt;
&lt;context-param&gt;
          &lt;param-name&gt;appName&lt;/param-name&gt;
          &lt;param-value&gt;some-arbitrary-string-unique-across-<b>all</b>-webapps&lt;/param-value&gt;
&lt;/context-param&gt;

&lt;listener&gt;
        &lt;listener-class&gt;fc.web.servlet.WebApp&lt;/listener-class&gt;
&lt;/listener&gt;
</pre>
</blockquote>
<p>
<font size="+2"><b>Important</b></font>:<u>If this class is used <font
<i>and</i> its initialization is not successful</u>, <font size="+2">it tries to
shut down the entire servlet JVM</font> by calling <tt>System.exit</tt>. (The
idea being it's better to fail early and safely then continue beyond this
point). 
<p>
If used, this class requires the following context configuration 
parameter:
<blockquote>
<ul>
<li><tt>configfile</tt>: Path/name of the application configuration
file. If the path starts with a '/', it is an absolute file system path.
Otherwise, it is relative to this context root's WEB-INF directory.</li>
<li><tt>appName</tt>: Some arbitrary (but unique) name associated with this 
webapp</li>
</ul>
</blockquote>
<p>
This class can also be subclassed to initialize/contain website specific data
and background/helper processing threads. Alternatively, along with this class
as-is, additional independent site-specific ServletContextListener classes can
be created and used as necessary.

@author hursh jain
*/
public class WebApp implements ServletContextListener
{
//IMPL NOTE: synchronize stuff that is set here with AdminServlet
/*
Contains all the {@link fc.dbo.ConnectionMgr ConnectionManagers} 
for this application.
*/
public Map						connectionManagers = new HashMap();
public ConnectionMgr			defaultConnectionManager;
public long					  	default_dbcache_time = MemoryCache.TWO_MIN;
public ThreadLocalCalendar	  	default_tlcal;
public ThreadLocalDateFormat	default_tldf;
public ThreadLocalNumberFormat	default_tlnf;
public ThreadLocalRandom		default_tlrand;
public Map					  	appMap   = new Hashtable(); //hashtable is sync'ed
public Map					  	tlcalMap = new Hashtable(); //ht is sync'ed
public Map					  	tldfMap  = new Hashtable(); //ht is sync'ed
public Map					  	tlnfMap  = new Hashtable(); //ht is sync'ed
public Map					  	tlrandMap = new Hashtable(); //ht is sync'ed
public Map					  	tlMap 	  = new Hashtable(); //ht is sync'ed

/**
A {@link Log} object. Servlets typically create their own
loggers (with servlet specific logging levels) but can alternatively
use this default appLog. This appLog is used by non-servlet classes
such as this class itself, various listeners etc.
*/
public 		Log				appLog;
public 		PropertyMgr 	propertyMgr;
	   		long			cache_time; //in ms
	   		
/** The required 'appName' parameter read from the config file (web.xml) **/
protected	String			appName;

/** 
A (initially empty) Map that servlets can use to store a reference to
themselves. This is required because the servlet API has deprecated a
similar API call (the servlet API authors are brain damaged 'tards).
*/
public 			Map					allServletsMap 	= new HashMap();
private static 	Map<String, WebApp> instances 		= new HashMap();
/** this is for WebApps added externally/manually by a servlet via addWebApp. When
the context/webapp under which that servlet is running is destroyed, that manually
added webapp is also automatically destroyed */
private static 	Set 			 	autoDestroySet = new HashSet();


/** 
A no-arg constructor public such that the servlet container can instantiate this class.
*/
public WebApp() { }

/*
Returns an instance of the webapp configured for this application (or <tt>null</tt> 
if not yet configured (via the servlet context initialization) or not found.
*/
public static WebApp getInstance(String appName)
	{
	synchronized (WebApp.class) 
		{
		return instances.get(appName);
		}
	}
	
/**
Basic implementation of the web application cleanup upon context creation.

If this method is subclassed, the subclassed method must also invoke this
method via <tt>super.contextInitialized</tt>. This call should typically
be at the beginning of the subclassed method, which allows this implementation
to create connections, logs etc, which can then be used by the subclass to
finish it's further initialization as needed.
*/
public void contextInitialized(ServletContextEvent sce)
	{
	ServletContext context = sce.getServletContext();
	try {
		String appName = WebUtil.getRequiredParam(context, "appName");
		String sconf = WebUtil.getRequiredParam(context, "configfile");	
		
		addWebAppImpl(context, appName, sconf, this);	
		}
	catch (ServletException e) {
		System.err.println("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
		System.err.println("**** ERROR: exception in " + getClass().getName() + " *****");
		System.err.println("**** Shutting down the Web Server ****");
		System.err.println("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
		System.err.flush();
		e.printStackTrace(System.err); //send email to someone here ?

		//There really isn't a good way to stop just this app from working (and
		//keep other webapps/hosts alive). There isn't any way to throw a 
		//UnavailableException from here, those exceptions are at each individual
		//servlet level, not webapp/context level. Freaking dumb. Can use flags
		//and a catch all filter but too much work. Shut this fucker down.
		
		System.exit(1);
		}
	} //~WebApp initialized


/**
Basic implementation of the web application cleanup upon context destruction.

If this method is subclassed, the subclassed method must also invoke this
method via <tt>super.contextInitialized</tt>. This invokation should typically
be at the <b>end</b> of the subclassed method (which allows the subclass to
use connections etc., before they are closed by this superclass method).
*/
public void contextDestroyed(ServletContextEvent sce)
	{
	appLog.info("WebApp [", appName, "] closing at: ", new java.util.Date());
	
	cleanup(appName);

	ServletContext context = sce.getServletContext();
	String appName2 = null;
	try {
		appName2 = WebUtil.getRequiredParam(context, "appName");
		if (! this.appName.equals(appName2)) {
			System.err.println("Weird, appName (" + appName + ") and context.appName(" + appName2 + ") were different");
			cleanup(appName2);
			}
		}
	catch (Exception e) {
		IOUtil.throwableToString(e);
		}

	//auto destroy (manually added webapps)
	Iterator it = autoDestroySet.iterator();
	while (it.hasNext()) {
		String name = (String) it.next();
		cleanup(name);
		it.remove();
		}
	}
	
private void cleanup(String appName)
	{
	WebApp app = (WebApp) instances.remove(appName);

	if (app == null) {
		return;
		}
		
	Iterator it = app.connectionManagers.values().iterator();
	
	while (it.hasNext()) {
		ConnectionMgr cmgr = (ConnectionMgr) it.next();
		cmgr.close();
		}		
	}


/**
Add a new WebApp instance manually. A WebApp is typically specified as a context
listener, in a particular <i>web.xml</i>, and configures itself as representing
that context wide configuration (common to all servlets and pages running inside
that context). There is only 1 instance of the WebApp class that is instantiated
per context by the servlet container.
<p>
However, for REST services and other API's, it is useful to have several API
versions, such as <tt>/rest/v1/</tt>, <tt>/rest/v2/</tt>, etc., all of which are
tied to seperate servlets in the <b>same</b> webapp. We could also create
separate webapps inside separate folders, such as <tt>/w3root/rest/v1</tt>,
<tt>/w3root/rest/v2</tt>, each of which would have their own web.xml (and hence
separate configuration).
<p>
However, we sometimes need to access a particular REST api's <b>classes</b> directly
from the "root" document webapp (which is running molly pages, etc). If the REST api's
are in separate contexts, the root webapp (a different webapp from all the other
REST api versioned webapps) cannot access those api classes directly. This becomes
a hassle if we want to use our API directly to show results on a molly web page. 
<p>
So we use only <b>one</b> webapp, with separate servlets in that webapp. Example: 
<tt>RESTServletV1</tt>, <tt>RESTServletV2</tt>, etc., to handle and serve different 
version of the API.
<p>
So then, each of these servlets have to be configured as well with connection pools,
loggers, etc, each of them specific to a particular servlet/api. This can be done
via servlet init parameters, but since the whole point of WebApp is to set up this
configuration easily, it is also useful to create a WebApp instance <i>per 
servlet instance</i> in the same webapp. 
<p>
Servlets can then call this method with different appNames (unique to each servlet)
and different configuration files (again unique to each servlet). When the context
is destroyed, all these manually added webapps will also be automatically closed.
<p>
This class must be specified as a listener in web.xml (even if the context wide
config file has no configuration data, use an empty config file in that case).
<p>
PS: If this makes your head hurt, you are in good company. My head hurts too.

@param  context		the servlet context the calling servlet is running in
@param	appName		the name of the WebApp to associate with the calling servlet
@param	conf		the configuration file for the WebApp
*/
public static WebApp addWebApp(ServletContext context, String appName, String sconf)
	{
	WebApp app = new WebApp();
	addWebAppImpl(context, appName, sconf, app);
	autoDestroySet.add(appName);
	
	return app;
	}

private static void addWebAppImpl(ServletContext context, 
		String appName, String sconf, WebApp app)
	{
	java.util.Date now = Calendar.getInstance().getTime();
	System.out.println("*******************************************************************");
	System.out.println("fc.web.servlet.WebApp: starting initialization on: " + now);
	System.out.println("WebApp running at: " + context.getRealPath(""));
	System.out.println("*******************************************************************");
	try {
		if (instances.containsKey(appName)) 
			{
			System.err.println("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
			String err = "A webapp with name[" 
				+ appName + "] at context [" 
				+ context.getRealPath(context.getContextPath()) + "] already exists..";
			System.err.println(err);
			//this *should* prevent any servlet in this context from working.
			//gentler than System.exit()
			throw new RuntimeException(err);
			}
			
		app.appName = appName;	
			
		Log appLog = Log.get(appName);
		app.appLog = appLog;
		
		File conf = new File(sconf);
		System.out.println("-> WebApp: Configuration file: " + conf.getPath());
		
		if (! conf.isAbsolute()) {
			conf = new File(context.getRealPath("/WEB-INF"), conf.getPath());
			}
		
		PropertyMgr propertyMgr = new FilePropertyMgr(conf);
		app.propertyMgr = propertyMgr;
		
	    System.out.println("-> WebApp: " +  propertyMgr);
		
		String level = propertyMgr.get("log.level", null);
		if (level != null)  {
			appLog.setLevel(level);
			Log.setDefaultLevel(level);
			}
			
		ConnectionMgr defaultCmgr = null;

		String dbdefault_str = propertyMgr.get("db.default");
		app.appLog.info("WebApp: Default database = ", 
				dbdefault_str == null ? "Not specified" : dbdefault_str);
		
		String dblist_str = propertyMgr.get("db.list");
		app.appLog.info("WebApp: All databases: ", dblist_str == null ?
										"None specified" : dblist_str);
		
		if (dblist_str == null) {
			if (dbdefault_str != null) {
				throw new IllegalArgumentException("Since a default database was specified, a dblist must also be specified (but was null)");
				}
			}
		else{	
			String[] cmgrnames = dblist_str.split(",");		
			for (String dbname : cmgrnames) 
				{
				dbname = dbname.trim();
				String poolsize = propertyMgr.get(dbname + ".pool.size");		 
				String jdbc_url = propertyMgr.get(dbname +  ".jdbc.url");
				String jdbc_driver = propertyMgr.get(dbname +  ".jdbc.driver");
				String jdbc_user = propertyMgr.get(dbname +  ".jdbc.user");
				String jdbc_password = propertyMgr.get(dbname +  ".jdbc.password");
				String jdbc_catalog = propertyMgr.get(dbname +  ".jdbc.catalog", "");
			
				ConnectionMgr cmgr = new PooledConnectionMgr(
					  jdbc_url, jdbc_driver, jdbc_user, jdbc_password, 
					  jdbc_catalog, Integer.parseInt(poolsize));
			
				app.connectionManagers.put(dbname, cmgr);
				if (dbname.equalsIgnoreCase(dbdefault_str))
					app.defaultConnectionManager = cmgr;
				}
 			}
		
		String cache_time_str = propertyMgr.get("db.cache_time_seconds");
	
		if (cache_time_str != null) {
			app.cache_time = Long.parseLong(cache_time_str) * 1000;
			}
		else{
			app.cache_time = app.default_dbcache_time;
			}
	
		appLog.info("WebApp: db.cache_time = ", app.cache_time/1000, " seconds"); 
	
		QueryUtil.init(appLog);
		instances.put(appName, app);
		
		System.out.println("===> WebApp finished [success]");
		System.out.println("*******************************************************************\n");
		}
	catch (Exception e) 
		{
		System.err.println("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
		System.err.println("**** ERROR: exception in " + app.getClass().getName() + " *****");
		System.err.println("**** Shutting down the Web Server ****");
		System.err.println("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
		System.err.flush();
		e.printStackTrace(System.err); //send email to someone here ?

		//There really isn't a good way to stop just this app from working (and
		//keep other webapps/hosts alive). There isn't any way to throw a 
		//UnavailableException from here, those exceptions are at each individual
		//servlet level, not webapp/context level. Freaking dumb. Can use flags
		//and a catch all filter but too much work. Shut this fucker down.
		
		System.exit(1);
		}
	}


/** 
Returns the property manager associated with this WebApp (properties of the app
configuration file can be read via this property mgr)
*/
public PropertyMgr getPropertyMgr()
	{
	return propertyMgr;	
	}
				
/** 
Returns the connection manager corresponding to the database
name. (specified in the dblist property of app.conf). 

@throws IllegalArgumentException  if the specified database is not found
*/
public ConnectionMgr getConnectionMgr(String databasename)
	{
	ConnectionMgr cm = (ConnectionMgr) connectionManagers.get(databasename);
	if (cm == null) {
		throw new IllegalArgumentException("The specified connectionManager [" + databasename + "] does not exist or has not been initialized.");
		}
	return cm;	
	}
	
/** 
Returns the connection manager corresponding to the default database
name. (specified in the dbdefault property of app.conf). Returns <tt>null</tt>
if no default database has been initialized.
*/
public ConnectionMgr getConnectionMgr()
	{
	return defaultConnectionManager;
	}

/*
returns a connection from the connection pool to the database specified.
blocks until a connection is available.

@param databasename  database from the dblist property of app.conf
@throws IllegalArgumentException  if the specified database is not found
*/
public Connection getConnection(String databasename) throws SQLException
	{
	return getConnectionMgr(databasename).getConnection();
	}
	
/*
returns a connection from the connection pool to the default database.
blocks until a connection is available.

*/
public Connection getConnection() throws SQLException
	{
	if (defaultConnectionManager == null) {
		throw new SQLException("The ConnectionManager in " + WebApp.class + " does not have \"default\" DB. Specify a DB name to get a connection to that DB.");
		}
	return defaultConnectionManager.getConnection();
	}
	
/*
Returns the default application log.
*/
public Log getAppLog()
	{
	return appLog;
	}

/**
Convenience method to return a form stored previously via the {@link
putForm} method. Returns <tt>null</tt> if no form with the specified name
was found.
*/
public Form getForm(String name)
	{
	return (Form) appMap.get(name);
	}

public void putForm(Form f)
	{
	appMap.put(f.getName(), f);
	}
	
public void removeForm(String name)
	{
	if (appMap.containsKey(name))
		appMap.remove(name);
	}


/**
Convenience method to get the pre-created application cache. This
can be used for caching database results as necessary. Each entry
in the cache can be stored for entry-specific time-to-live (see {@link Cache)
but if no entry-specific-time-to-live is specified, then entries are
cached for a default time of 5 minutes. Of course, cached entries should
always be invalidated sooner whenever the database is modified.
*/
public Cache getDBCache()
	{
	Object obj = appMap.get("_dbcache");
	
	if (obj != null) 
		return (Cache) obj;
	
	Cache cache = null;
	
	synchronized (WebApp.class) {  //2 or more threads could be running in a page
		cache = new MemoryCache(
				Log.get("fc.util.cache"), "_dbcache", cache_time);
		appMap.put("_dbcache", cache);
		}
		
	return cache;	
	}

/** 
Returns the {@link ThreadLocalDateFormat} object corresponding to the specified
name (a new ThreadLocalDateFormat is created if it does not exist).
*/
public ThreadLocalDateFormat getThreadLocalDateFormat(String name)
	{
	ThreadLocalDateFormat tldf = (ThreadLocalDateFormat) tldfMap.get(name);
	
	if (tldf == null) {
		tldf = new ThreadLocalDateFormat();
		tldfMap.put(name, tldf);
		}
		
	return tldf;
	}

/** 
Returns the default threadLocalDateFormat object (a new 
ThreadLocalDateFormat is created if it does not exist).
*/
public ThreadLocalDateFormat getThreadLocalDateFormat()
	{
	if (default_tldf == null) {
		default_tldf = new ThreadLocalDateFormat();
		}
		
	return default_tldf;
	}
	


/** 
Returns the ThreadLocalNumberFormat object corresponding to the specified
name (a new ThreadLocalNumberFormat is created if it does not exist).
*/
public ThreadLocalNumberFormat getThreadLocalNumberFormat(String name)
	{
	ThreadLocalNumberFormat tlnf = (ThreadLocalNumberFormat) tlnfMap.get(name);
	
	if (tlnf == null) {
		tlnf = new ThreadLocalNumberFormat();
		tlnfMap.put(name, tlnf);
		}
		
	return tlnf;
	}
	
/** 
Returns the default {@link ThreadLocalNumberFormat} object (a new
ThreadLocalNumberFormat is created if it does not exist).
*/
public ThreadLocalNumberFormat getThreadLocalNumberFormat()
	{
	if (default_tlnf == null) {
		default_tlnf = new ThreadLocalNumberFormat();
		}
		
	return default_tlnf;
	}
	
/** 
Returns the {@link ThreadLocalCalendar} object corresponding to the
specified name (a new ThreadLocalCalendar is created if it does
not exist).
*/
public ThreadLocalCalendar getThreadLocalCalendar(String name)
	{
	ThreadLocalCalendar tlcal = (ThreadLocalCalendar) tlcalMap.get(name);
	
	if (tlcal == null) {
		tlcal = new ThreadLocalCalendar();
		tlcalMap.put(name, tlcal);
		}
		
	return tlcal;
	}
	
/** 
Returns the default ThreadLocalCalendar object (a new ThreadLocalCalendar
is created if it does not exist).
*/
public ThreadLocalCalendar getThreadLocalCalendar()
	{
	if (default_tlcal == null) {
		default_tlcal = new ThreadLocalCalendar();
		}
		
	return default_tlcal;
	}

/** 
Returns the {@link ThreadLocalRandom} object corresponding to the
specified name (a new ThreadLocalRandom is created if it does
not exist).
*/
public ThreadLocalRandom getThreadLocalRandom(String name)
	{
	ThreadLocalRandom tlrand = (ThreadLocalRandom) tlrandMap.get(name);
	
	if (tlrand == null) {
		tlrand = new ThreadLocalRandom();
		tlrandMap.put(name, tlrand);
		}
		
	return tlrand;
	}
	
/** 
Returns the default ThreadLocalRandom object (a new ThreadLocalRandom
is created if it does not exist).
*/
public ThreadLocalRandom getThreadLocalRandom()
	{
	if (default_tlrand == null) {
		default_tlrand = new ThreadLocalRandom();
		}
		
	return default_tlrand;
	}

/** 
Returns the {@link ThreadLocalObject} corresponding to the
specified name (a new ThreadLocalObject is created if it does
not exist).
*/
public ThreadLocalObject getThreadLocalObject(String name)
	{
	ThreadLocalObject tlo = (ThreadLocalObject) tlMap.get(name);
	
	if (tlo == null) {
		tlo = new ThreadLocalObject();
		tlMap.put(name, tlo);
		}
		
	return tlo;
	}
	

/**
Returns the specified object from the global application map or 
<tt>null</tt> if the object was not found.
*/	
public Object get(Object key)
	{
	return appMap.get(key);
	}

/**
Puts the specified key/object into the global application map.
*/	
public void put(Object key, Object val)
	{
	appMap.put(key, val);
	}

public String toString() {
	return new ToString(this).reflect().render();
	}
}
