// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.servlet;

import java.util.*;
import java.util.logging.*;
import java.io.*;
import javax.servlet.*;
import javax.servlet.http.*;
import java.net.*;

import fc.io.*;

/** 
Misc web related utility functions 

@author hursh jain
**/
public final class WebUtil 
{
final static Log log = Log.get("fc.web.servlet");

//--------------------- getParameters ----------------------

/** 
Gets the specified required initialization parameter from
the servlet context. If this param is not found, throws a
permanent {@link javax.servlet.UnavailableException}.
**/
public static final String getRequiredParam
	(ServletContext context, String name) 
throws ServletException
 	{
 	String param = context.getInitParameter(name);		

	if (param == null) {
		String error = "Required Servlet Init Parameter: '" + name + "' was not found";
		log.error(error);
 		throw new UnavailableException(error);
		}

 	return param;	
 	}

/** 
Gets the specified initialization parameter from the servlet context. If 
this param is not found, then the specified backup value is returned.
**/
public static final String getParam(
	ServletContext context, String name, String backup) 
 	{
 	String param = context.getInitParameter(name);		
 	if (param != null)
 		return param;	
 	else
		return backup;
 	}

/** 
Gets the specified initialization parameter for the specified servlet
(via it's ServletConfig). Note, servlet init parameters are servlet
specific whereas context parameters are shared by all servlets within
that context. If this param is not found, throws a permanent {@link
javax.servlet.UnavailableException}.
**/
public static final String getRequiredParam(Servlet servlet, String name) 
throws ServletException
 	{
	ServletConfig config = servlet.getServletConfig();
	String param = config.getInitParameter(name);		

	if (param == null) {
		String error = "Required Servlet Init Parameter: '" + name + "' for servlet '" + config.getServletName() + "' was not found";
		log.error(error);
 		throw new UnavailableException(error);
		}

 	return param;	
 	}

/** 
Gets the specified initialization parameter for the specified servlet
(via it's ServletConfig). Note, servlet init parameters are servlet
specific whereas context parameters are shared by all servlets within
that context. If this param is not found, then the specified backup value
is returned.
**/
public static final String getParam(
	Servlet servlet, String name, String backup) 
 	{
	ServletConfig config = servlet.getServletConfig();
 	String param = config.getInitParameter(name);		
 	if (param != null)
 		return param;	
 	else
		return backup;
 	}

/** 
Gets the specified required parameter from the request
object. If this param is not found, throws a ServletException
**/
public static final String getRequiredParam(
						HttpServletRequest request, String name) 
throws ServletException
	{
	String param = request.getParameter(name);		

	if (param == null) {
		String error = "Required Parameter: '" + name + "' was not found";
		log.error(error);
		throw new ServletException(error);
		}

	return param;	
	}

/** 
Gets the specified parameter from the request object. 
If this param is not found, returns the backup value.
**/
public static final String getParam(
	HttpServletRequest request, String name, String backup) 
	{
	String param = request.getParameter(name);		
	if (param != null)
 		return param;	
 	else
		return backup;
	}

/** 
The returned value will be <tt>true</tt> if the specified parameter
is present in the request and is a non-null non-empty string (<b>of
any value</b>). This is useful for radioboxes and checkboxes.
**/
public static final boolean isSelected(
	HttpServletRequest request, String name) 
	{
	String param = request.getParameter(name);		
	if (param == null || param.equals(""))
		return false; 			
		
	return true;
	}

/** 
Convenience method that returns the specified parameter as a boolean 
value. The returned value will be converted via the
{@link Boolean#parseBoolean(String)} (for <tt>true</tt>, the 
value should be non-null and equal ignore case to "true").
**/
public static final boolean getBooleanParameter(
	HttpServletRequest request, String name)
	{
	String param = request.getParameter(name);		
	return Boolean.parseBoolean(param);
	}


//--------------------- getAttributes ----------------------
	
/** 
Gets the specified required attribute from the servlet context. If 
this attribute is not found, throws a permanent 
{@link javax.servlet.UnavailableException}.
**/
public static final Object getRequiredAttribute(
	ServletContext context, String name) 
throws ServletException
 	{
 	Object param = context.getAttribute(name);		

	if (param == null) {
		String error = "Required context attribute: '" + name + "' was not found";
		log.error(error);
		throw new UnavailableException(error);	
		}

 	return param;	
 	}

/** 
Gets the specified attribute from the servlet context. If this attribute
is not found, then the specified backup value is returned.
**/
public static final Object getAttribute(
	ServletContext context, String name, String backup) 
throws ServletException
 	{
 	Object param = context.getAttribute(name);		
	
 	if (param != null) 		
		return param;	
 	else
		return backup;
 	}

/** 
Gets the specified required attribute from the request object. 
If this param is not found, throws a ServletException.
**/
public static final Object getRequiredAttribute(
						HttpServletRequest request, String name) 
throws ServletException
	{
	Object param = request.getAttribute(name);		
	
	if (param == null) {
		String error = "Required Parameter: '" + name + "' was not found";
		log.error(error);
		throw new ServletException(error);	
		}

	return param;	
	}

/** 
Gets the specified attribute from the request object. 
If this attribute is not found, returns the backup value.
**/
public static final Object getAttribute(
	HttpServletRequest request, String name, String backup) 
throws ServletException
	{
	Object param = request.getAttribute(name);		
	if (param != null)
 		return param;	
 	else
		return backup;
	}

//----------------------------------------------------------

/** 
Returns a HttpSession attribute as a String or null if the attribute
was absent. 

@param	name	the name of the session attribute
@throws ClassCastException 		if the attribute was not a String
@throws NullPointerException 	if the specified session or name parameters
								were null
**/	
public static final String getSessionString(HttpSession session, String name) 
	{
	Object obj = session.getAttribute(name);
	if (obj != null)
		return (String) obj;
	else
		return null;
	}

/**
Returns the cookie with the specified name or <tt>null</tt> if no cookie
was found.
*/
public static Cookie getCookie(HttpServletRequest req, String cookieName)
	{
	Cookie[] cookies = req.getCookies();
	
	if (cookies == null)
		return null;
		
	for (int n = 0; n < cookies.length; n++) 
		{
		if (cookies[n].getName().equals(cookieName))
			{
			return cookies[n];
			}
		}
	return null;
	}
	
/** 
Redirects the request (server side redirect) to the specified relative URL
via the {@link javax.servlet.ServletRequest#getRequestDispatcher}. See
{@link javax.servlet.RequestDispatcher#forward} for restrictions while
forwarding to another page (in particular, the response must not have been
committed before calling this method).
<p>
The calling code <b>must</b> have a <tt>return</tt> statement immediately
after calling this method, otherwise RequestDispatching will probably not
work as intended.

@param	req		the current request
@param	res		the current response
@param	URL		The pathname specified may be relative, although it 
				cannot extend outside the current servlet context. If the
				path begins with a "/" it is interpreted as starting from
				the root of the current context. (i.e., the webapp context,
				if present, does not need to be specified).
**/
public static final void forward ( HttpServletRequest req, 
								   HttpServletResponse res,
								   String URL)
throws ServletException, IOException
	{
	RequestDispatcher rd = req.getRequestDispatcher(URL);
	if (log.canLog(Log.DEBUG)) {
		StringBuffer buf = getRequestURL(req);
		log.bug(buf, "forwarding to page: ", URL);
		}
	rd.forward(req, res);	
	}

/** 
Redirects the browser to the specified URL via a client side redirect URL.
Automatically creates a full URL <u>(including the webapp context path)</u>
suitable for this purpose. This method is a thin wrapper around {@link
javax.servlet.http.HttpServletResponse#sendRedirect Response.sentRedirect} 
<p>
Session information is preserved if using container provided URL-based sessions. 
<p>
<u>If the response has already been committed, this method throws an
<tt>IllegalStateException</tt></u>. After using this method, the response should be
considered to be committed and should not be written to.

@param	req			the current request
@param	res			the current response
@param	location	See 
					{@link javax.servlet.http.HttpResponse#sendRedirect}.
					Essentially, the location can be relative to the
					specified request's URI or relative to the 
					context root if it contains a leading '/'
**/
public static final void clientRedirect(
	HttpServletRequest req, HttpServletResponse res, String location)
throws ServletException, IOException
	{
	String redirectURL = res.encodeRedirectURL(
							absolutePath(req, location));
	if (log.canLog(Log.DEBUG)) {
		StringBuffer buf = getRequestURL(req);
		log.bug(buf, "redirecting to page: ", redirectURL);
		}		
	res.sendRedirect(redirectURL);
	}

/** 
Creates an absolute path starting from the web document root directory. It
does so by prepending the webapp context path (if any) of the specified
request to the specified path. The path is <b>not</b> URL encoded with
session informtion (see {@link #absoluteEncPath}).
<p>
This method should be used for all HTML links (in html pages and forms) 
and also for client side redirects. This ensures the ability to move the calling code 
to a different web app (other than the default root ("/") web app). If
it's certain that there will always only be 1 web app (the default)
then calling this method is not necessary. 
<p>
Note also that server-side redirects and includes (with RequestDispatcher)
should *not* use this method since all such server side redirects etc
automatically work as expected inside the web app context and are always
relative to the web app context itself.

@param	req		the current HttpServletRequest
@param	path	an absolute path starting from the root of
				the context associated with the request. 
				If the current webapp (context) is <tt>/foo</tt> and
				the specified path is <tt>/bar</tt>, then this
				method will return <tt>/foo/bar</tt>
**/
public static final String absolutePath(HttpServletRequest req, String path) 
	{
	return req.getContextPath() + path;
	}

/** 
URL encodes and returns the path obtained via {@link #absolutePath}. 
**/
public static final String absoluteEncPath(
	HttpServletRequest req, HttpServletResponse res, String path) 
	{
	return res.encodeURL(absolutePath(req, path));
	}
	
/** 
@return the full requested URL including any query parameters.	
		(the {@link 
		javax.servlet.Http.HttpServletRequest.getRequestURL() RequestURL}
		method in HttpServletRequest does not return the query part of the URL)
**/
public static final StringBuffer getRequestURL(HttpServletRequest req) 
	{
	StringBuffer buf = 	req.getRequestURL();
	String qs = req.getQueryString();
	if (qs !=null) {
		buf.append('?');
		buf.append(qs);
		}
	return buf;
	}
	
	
//== HTML Form related ==========================

private static final String confirm = "confirm";
private static final String cancel = "cancel";

/** 
Returns <tt>true</tt> only if a request parameter with the
name <tt>confirm</tt> is seen. Forms can set a submit button
with this name to signal a confirm request.
**/
public static final boolean requestCancelled(HttpServletRequest req) {
	String str = req.getParameter(cancel);
	return (str != null); //testing for key not value
	}

/** 
Returns <tt>true</tt> only if a request parameter with the
name <tt>cancel</tt> is seen. Forms can set a submit button
with this name to signal a cancel request.
**/
public static final boolean requestConfirmed(HttpServletRequest req) {
	String str = req.getParameter(confirm);
	return (str != null); //testing for key not value
	}
	
/** 
writes all request params received in the specied request to
<tt>System.out</tt>. Useful for debugging.
**/
public static final void debugRequestParams(HttpServletRequest req) 
	{
	debugRequestParams(req, System.out);
	}
	

/** 
writes all request params received in the specied request to
the specified output stream. Useful for debugging.
**/
public static final void debugRequestParams(HttpServletRequest req, OutputStream out) 
	{
	System.out.print("Request params/");
	System.out.print(Thread.currentThread().getName());
	System.out.print(" ");
	
	Iterator i = req.getParameterMap().entrySet().iterator();
	while (i.hasNext()) {
		Map.Entry me = (Map.Entry) i.next();
		System.out.print(me.getKey());
		System.out.print(":");
		System.out.print(Arrays.toString((String[])me.getValue()));
		
		if (i.hasNext()) 
			System.out.print(", ");
		else 
			System.out.println();
		}
	}


/** 
set expire headers to cover all cases
**/
public static final void setExpireHeaders(HttpServletResponse res) 
	{
	// Set standard HTTP/1.1 no-cache headers.
	res.setHeader("Cache-Control", "no-store, no-cache, must-revalidate");
	// Set IE extended HTTP/1.1 no-cache headers (use addHeader).
	res.addHeader("Cache-Control", "post-check=0, pre-check=0");
	// Set standard HTTP/1.0 no-cache header.
	res.setHeader("Pragma", "no-cache");
	//expire in past
	res.setHeader("Expires", "-1");
	}

/*
Parse a query parameter string (useful for things like websocket and custom
servers. The query param should be url form encoded and is decoded using
UTF-8 (via {@link java.net.URLDecoder}). Returns a map containing each
<tt>name:value</tt> pair
*/
public static Map parseQuery(String query) throws UnsupportedEncodingException 
	{
    Map m = new HashMap();
    String[] params = query.split("&");
    for (int n = 0; n < params.length; n++) {
    	String param = params[n]; 
    	int i = param.indexOf("=");
        m.put(
        	URLDecoder.decode(param.substring(0, i), "UTF-8"), 		
        	URLDecoder.decode(param.substring(i + 1), "UTF-8"));
    	}
    return m;
	}

/**
Returns true if the client is a mobile device 
*/
public static boolean isMobileClient(HttpServletRequest req)
	{
	//https://developer.mozilla.org/en-US/docs/Web/HTTP/Browser_detection_using_the_user_agent

	//name is case insensitve (in servlet and http spec), different browsers send different cases so that's good
	String user_agent = req.getHeader("user-agent"); 
	return user_agent != null && user_agent.indexOf("obile") > 0; //Mobile or mobile 
	}

/**
Returns true if the client is an iOS mobile device.
*/
public static boolean isIOSClient(HttpServletRequest req)
	{
	//https://developer.mozilla.org/en-US/docs/Web/HTTP/Browser_detection_using_the_user_agent

	//name is case insensitve (in servlet and http spec), different browsers send different cases so that's good
	String user_agent = req.getHeader("user-agent"); 
	return user_agent != null && user_agent.indexOf("obile") > 0 && user_agent.indexOf("afari") > 0; 
	}
	
}          //~class WebUtil

