// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.simpleforms;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.*;

import fc.io.*;

/**
Convenience class to store arbitrary form validation errors
and messages. This class should be instantiated per request 
as needed (when there are form validation errors). 
<p>
Note: This class is not thread-safe but that's not a concern
since only each seperate user request is handled by at 
most 1 thread.

@author hursh jain
**/
public final class Errors
{
Log		log = Log.getDefault();

List	formErrors;
Map		fieldErrors;
List	formWarnings;
Map		fieldWarnings;

/**
Adds a form level error, typically associated with the form
itself and/or multiple fields as a group.
*/
public void addFormError(String msg)
	{
	if (formErrors == null)
		formErrors = new ArrayList();
	
	formErrors.add(msg);
	}

/**
Adds a field validation error

@param	fieldName	the name of the field
@param	msg 		some error object, typically a string but
					can be a list of strings (for example) if
					there is more than 1 validation error for
					this field
*/
public void addFieldError(String fieldName, Object msg)
	{
	if (fieldErrors == null)
		fieldErrors = new HashMap();
	
	Object old = fieldErrors.put(fieldName, msg);
	if (old != null) {
		log.warn("Use a list to add more than 1 message. I am currently over-writing the previous error message for field: " + fieldName + " [old msg=" + old + "] [new msg=" + msg + "]");
		}
	}

/**
Adds an arbitrary warning message generated as part of form processing
*/
public void addFormWarning(String msg)
	{
	if (formWarnings == null)
		formWarnings = new ArrayList();
	
	formWarnings.add(msg);
	}

/**
Adds an arbitrary warning message generated as part of form 
processing. This warning is associated with the specified field.
*/
public void addFieldWarning(String fieldname, Object msg)
	{
	if (fieldWarnings == null)
		fieldWarnings = new HashMap();
	
	Object old = fieldWarnings.put(fieldname, msg);
	if (old != null) {
		log.warn("Use a list to add more than 1 warning. I am currently over-writing the previous warning message for field: " + fieldname + " [old msg=" + old + "] [new msg=" + msg + "]");
		}
	}

/**
Returns a list of all form errors or null if no errors are present.
*/
public List getFormErrors() {
	return formErrors;
	}

/**
Returns the field error for the specified fieldname or null if no 
error was found.
*/
public Object getFieldError(String fieldName) 
	{
	if (fieldErrors == null)
		return null;
		
	return fieldErrors.get(fieldName);
	}

/**
Returns the list of all form-level warnings or <tt>null</tt> if no
warnings exist for the form.
*/
public List getFormWarnings() {
	return formWarnings;
	}

/**
Returns the warning for the specified field or <tt>null</tt> if
no warning exists.
*/
public Object getFieldWarning(String fieldname) {
	return fieldWarnings.get(fieldname);
	}

/**
Returns true if there are any form or field errors. (although
warnings are allowed)
*/
public boolean hasError() {
	return formErrors != null || fieldErrors != null;
	}
	
/**
Returns true if there are any warnings.
*/
public boolean hasWarning() {
	return formWarnings != null || fieldWarnings != null;
	}
	
/**
Returns true if there are any warnings for the specified <u>field</u>
*/
public boolean hasWarning(String fieldname) {
	return fieldWarnings != null && fieldWarnings.containsKey(fieldname);
	}

	
/**
Convenience method to render all the form errors (if present).
For more control, obtain the form errors and print them manually.
Invoking this method has the following effect:
<blockquote><pre>
String after = "&lt;br&gt;";
List list = error.getFormErrors();
if (list != null) 
	{
    out.write("&lt;div class='form-errmsg'&gt;");
    out.write("&lt;ul&gt;");
    for (int n = 0; n < list.size(); n++) {
    	out.write("&lt;li&gt;");
        out.write(String.valueOf(list.get(n)));
        out.write("&lt;/li&gt;");
        }
    out.write("&lt;/ul&gt;");
    out.write("&lt;/div&gt;\n");
    }
</pre></blockquote>
*/
public void renderFormErrors(Writer out) throws IOException
	{
	if (formErrors == null) {
		return;
		}

	if (formErrors == null)
		return;
		
	out.write("<div class='form-errmsg'>");
	out.write("\n<ul>");
	
	for (int n = 0; n < formErrors.size(); n++) {
		Object obj = formErrors.get(n);
		out.write("<li>");
		out.write(String.valueOf(obj));
		out.write("</li>\n");
		}

	out.write("</ul>\n");	
	out.write("</div>");
	}

/**
Convenience method to render a field error. Invoking this
method is a shorthand for saying (where <tt>error</tt> is
an instance of this class):
<blockquote><pre>
Object obj = error.getFieldError("some_field_name");
if (str != null) {
    out.write("&lt;span class='field-errmsg'&gt;");
    out.write (String.valueOf(obj));
    out.write("&lt;/span&gt;\n");
    out.write("&lt;br&gt;"); 
    }
</pre></blockquote>
The above is the same as:
<blockquote><pre>
error.<font color=blue>render</font>(out, "some_field_name");
</pre></blockquote>
Note: The object representing the error for the field is written as
is. Typically for strings, this works fine. However, for more complex
objects (like say a list holding more than 1 error for the same field),
the list is printed as-is. For more formatting options for complex
objects, obtain and print the error manually.
*/
public void render(Writer out, String fieldName) throws IOException
	{
	if (fieldErrors == null) {
		return;
		}

	Object obj = fieldErrors.get(fieldName);
	if (obj == null)
		return;

	out.write("<span class='field-errmsg'>");
	out.write(String.valueOf(obj));
	out.write("</span>");
	out.write("<br>");
	}


/**
Convenience method to render a field error. Invoking this
method is a shorthand for saying (where <tt>error</tt> is
an instance of this class):
<blockquote><pre>
Object obj = error.getFieldError("some_field_name");
if (str != null) {
    out.write("&lt;span class='field-errmsg'&gt;");
    out.write (String.valueOf(obj));
    out.write(<font color=blue>"inside"</font>);
    out.write("&lt;/span&gt;\n");
    out.write(<font color=blue>"outside"</font>); 
    }
</pre></blockquote>
The above is the same as (for example):
<blockquote><pre>
error.<font color=blue>render</font>(out, "some_field_name", <font color=blue>"inside"</font>, <font color=blue>"outside"</font>);
</pre></blockquote>
Note: The object representing the error for the field is written as
is. Typically for strings, this works fine. However, for more complex
objects (like say a list holding more than 1 error for the same field),
the list is printed as-is. For more formatting options for complex
objects, obtain and print the error manually.

@param inside		this string is written before the span tag is closed
@param outside		this string is written right after the span tag is closed.
*/
public void render(
 Writer out, String fieldName, String inside, String outside) 
 throws IOException
	{
	if (fieldErrors == null) {
		return;
		}

	Object obj = fieldErrors.get(fieldName);
	if (obj == null)
		return;

	out.write("<span class='field-errmsg'>");
	out.write(String.valueOf(obj));
	out.write(inside);
	out.write("</span>");
	out.write(outside);
	}



public static void main (String args[]) throws Exception
	{
	Errors e = new Errors();
	e.addFieldError("a", "some field error");
	List list = new ArrayList();
	list.add("item1");
	list.add("item2");
	e.addFieldError("b", list);
	System.out.println("Has warnings: [should be false]: " + e.hasWarning());
	System.out.println("Has errors: [should be true]:    " + e.hasError());
	e.addFormError("some form error");
	System.out.println("Has errors: [should be true]:    " + e.hasError());
	PrintWriter pw = new PrintWriter(System.out);
	pw.println("\n--------- form errors ---------");
	e.renderFormErrors(pw);
	pw.println("\n--------- field errors [a]--------");
	e.render(pw, "a");
	pw.println("\n--------- field errors [b]--------");
	e.render(pw, "b");
	pw.println("\n--------- field errors [c/non-existent]--------");
	e.render(pw, "c");
	pw.flush();
	}

}
