// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.simpleforms;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.*;

import fc.io.*;
import fc.util.*;

/**
Helps maintain form state. Form state needs to be maintained
when a form is submitted but needs to be redisplayed to the
user because of errors or incomplete fields. 
<p>
For example:
<blockquote><pre>
&lt;input name=<font color=green>foo</font> type='text' value='<font color=blue>[= State.text(req, "<font color=green>foo</font>")]'</font>&gt; 
</pre></blockquote>
Often we need to set default/initial values (say from a database) for 
pre-filling a form before it's shown to the user (for example, when
the user is editing some information that already exists in the database).
In the simple form API, this is done via invoking the {@link 
#set(HttpServletRequest, String, String) set(req, field_name, field_value)} 
method, which sets the initial value of the field in the specified
HttpServletRequest object (mimicking as if the user had entered those values
and submitted the form). Since various widgets maintain state by retrieveing
these values from the HttpServletRequest, these values will be shown 
to the user. For example:
<blockquote><pre>
String initial_value = databaseQuery.getUserName(); //for example
State.set(req, "<font color=green>foo</font>", initial_value);

...later in the page...
//this will show the initial value set above and also maintain state
//when the user changes the value and submits the form
&lt;input name=<font color=green>foo</font> type='text' value='<font color=blue>[= State.text(req, "<font color=green>foo</font>")]'</font>&gt; 
</pre></blockquote>

@author hursh jain
*/
public final class State
{
private static final boolean dbg 	= false;
				
//text should always be at least an empty string. However, can
//can be hacked. Also browser return no value for disabled fields
//we don't track field enable/disable status and to keep simpleforms
//simple, don't have a overloaded method that takes status as a
//param. so no hacklerts here.

//TEXT, TEXTAREA, PASSWORD, RADIO, CHECKBOX, HIDDEN, SELECT

// ------------------------------------------------------------

private static final String getParameter(
	final HttpServletRequest req, final String name)
	{
	if (req.getAttribute(cleared_key) == null) //use submitted data ?
		{
		final String val = req.getParameter(name);
		if (val != null)
			return val;
		}

	
	/*
	return (String) req.getAttribute(name);		//some value or null
	//below prevents cut-n-paste typos
	*/
	final Object val = req.getAttribute(name);
	try {
		return (String) val;
		}
	catch (ClassCastException e) {
		throw new RuntimeException("The value of this field [" + name + "] should be set ONLY once (and as a string). You cutting'n'pasting dude ?");
		}
		
	}
	
private static final String[] getParameterValues(
	final HttpServletRequest req, final String name)
	{
	if (req.getAttribute(cleared_key) == null) //use submitted data ?
		{
		final String[] val = req.getParameterValues(name);
		if (val != null) {
			return val;
			}
		}
		
	//returns null or String[]
	final Object obj = req.getAttribute(name);
	
	if (obj == null)
		return null;
		
	//in case we only added 1 string for a group
	if (obj instanceof String) { 
		String[] arr = new String[1];
		arr[0] = (String) obj;
		return arr;
		}
	
	final List list = (ArrayList) obj;
	return (String[]) list.toArray(str_arr_type);	
	}
	
private static final String[] str_arr_type = new String[] { };

//---------------------------------------------------------

/**
Returns the value of the submitted text area (or an empty string if
no value was submitted). Example:
<blockquote><pre>
&lt;input type=text name=<font color=green>foo</font> value='[=State.text(req, "<font color=green>foo</font>")]'&gt;
</pre></blockquote>
*/
public static final String text(
final HttpServletRequest req, final String name)
	{
	final String val = getParameter(req, name);
	if (dbg) System.out.println(name+"="+val);

	if (val != null) 
		return val;
	else
		return "";
	}

/**
Returns the value of the submitted text area (or an empty string if
no value was submitted). The returned value is trimmed of all
leading and trailing spaces.
<p>
The returned value replicates what the user entered and is not url
encoded. This is good when maintaining state.
<p>
Of course, when saving this out to the database, we should either 
strip naughty tags or save it as url encoded text. (otherwise, 
the door to cross-site scripting hacks is wide open). That's an 
orthogonal issue to maintaining form state via this method.
Example:
<blockquote><pre>
&lt;textarea name=<font color=green>foo</font>&gt;
[=State.textarea(req, "<font color=green>foo</font>")]
&lt;/textarea&gt;
</pre></blockquote>
*/
public static final String textarea(
final HttpServletRequest req, final String name)
	{
	//we need to trim(), otherwise
	// <textarea> NL
	// [state..   ] NL
	// </textarea>
	// The newline keep adding up and adding themselves to the state
	
	return text(req, name).trim();
	}


/**
Returns the value of the submitted password (or an empty string if
no value was submitted). Example:
<blockquote><pre>
&lt;input type=password name=<font color=green>foo</font> value='[=State.password(req, "<font color=green>foo</font>")]'&gt;
</pre></blockquote>
*/
public static final String password(
 final HttpServletRequest req, final String name)
	{
	return text(req, name);
	}

/**
Returns the submitted state of the submitted radio box. This radio
box should not be part of a radio group (multiple radio buttons with
the same name). For radio groups use the {@link #radiogroup
radiogroup} method. Example:
<blockquote><pre>
&lt;input type=radio name=<font color=green>foo</font> value='somevalue' [=State.radio(req, "<font color=green>foo</font>")]&gt;
</pre></blockquote>
<b>Note</b>: For radio boxes, the state maintainence value is the
presence/absence of the <tt>checked</tt> attribute. The value
returned by this method will be either an empty string or
<tt>checked</tt>. This should not be assigned to the 'value'
attribute (if present) of the radiobox. Rather, it should be a
stand-alone attribute. 

@param	name		the name of the radio element
@param	selected	<tt>true</tt> to select this radio box
					initially (the first time the form is shown to
					the user), <tt>false</tt> otherwise.
*/
public static final String radio(
	final HttpServletRequest req, final String name,
	final boolean selected)
	{
	final String val = getParameter(req, name);
	if (dbg) System.out.println(name+"="+val);
			
	if (val == null) 
		{
		if (selected)
			return "checked";
		else
			return "";
		}
	
	//value != null (any value)
	return "checked";
	}

/**
Calls {@link #radio(HttpServletRequest, String, boolean) radio(req, name, <b>false</b>)}
*/
public static final String radio(
	final HttpServletRequest req, final String name)
	{
	return radio(req, name, false);
	}


/**
Returns the submitted state of the specified radio box which is part
of a radio group (multiple radio buttons with the same name). For
non-group radio boxes, use the {@link #radio radio} method instead.
Example:
<blockquote><pre>
<b><u>Good:</u></b>
&lt;input type=radio name=<font color=green>foo</font> <u>value='<font color=blue>somevalue1</font>'</u> [=State.radiogroup(req, "<font color=green>foo</font>", <font color=blue>"somevalue1"</font>)]&gt;
&lt;input type=radio name=<font color=green>foo</font> <u>value='<font color=blue>somevalue2</font>'</u> [=State.radiogroup(req, "<font color=green>foo</font>", <font color=blue>"somevalue2"</font>)]&gt;

<b>Don't do this</b>
&lt;input type=radio name=<font color=green>foo</font> [=State.radiogroup(req, "<font color=green>foo</font>", <font color=blue>"on"</font>)]&gt;
</pre></blockquote>
This method takes a <code>value</code> parameter that specifies the
value associated with this particular radio box in the radio group.
Radios with no specific value are submitted with the value "on"
(case-insensitive) by the browser. (as seen in the second example above).
<u>This method requires that each radio in the radiogroup have a unique
and specified value tag.</u>
<p>
<b>Note</b>: For radio boxes, the state maintainence value is the
presence/absence of the <tt>checked</tt> attribute. The value
returned by this method will be either an empty string or
<tt>checked</tt>. This should not be assigned to the 'value'
attribute (if present) of the radiobox. Rather, it should be a
stand-alone attribute.
*/
public static final String radiogroup(
	final HttpServletRequest req, final String name, final String value)
	{
	final String[] val = getParameterValues(req, name);

	if (dbg) System.out.println(name+"="+Arrays.toString(val));
	
	//val could be the value or "on" if radio has no value tag
	if (val == null)
		return "";
	
	for (int n = 0; n < val.length; n++) 
		{
		if (val[n].equals(value) || val[n].equalsIgnoreCase("on")) {
			return "checked";
			}
		}
	
	return "";
	}

/*
Calls {@link checkbox(HttpServletRequest, String, boolean)} as
<tt>checkbox(HttpServletRequest, String, <b>false</b>)</tt>
*/
/*
..thot it was convenient but it's useless...the checkbox will
be checked every time if we say
  [=State.checkbox(req, "foo", true)],
because the true will override things even if the user unchecked
the box. so state is not maintained, too confusing...
..use the initial state (set) method to show initially checked boxes.

public static final String checkbox(
 final HttpServletRequest req, final String name)
	{
	return checkbox(req, name, false);
	}
*/

/**
Returns the checked state of the submitted checkbox. This radio box
should not be part of a radio group (multiple checkboxes with the
same name). For radio groups use the {@link #checkboxgroup
checkboxgroup} method.
Example:
<blockquote><pre>
&lt;input type=checkbox name=<font color=green>foo</font> value='somevalue' [=State.checkbox(req, "<font color=green>foo</font>")]&gt;
</pre></blockquote>
<b>Note</b>: For check boxes, the state maintainence value is the
presence/absence of the <tt>checked</tt> attribute. The value
returned by this method will be either an empty string or
<tt>checked</tt>. This should not be assigned to the 'value'
attribute (if present) of checkbox. Rather, it should be a
stand-alone attribute as shown above.

@param	name		the name of the radio element
@param	selected	<tt>true</tt> to select this checkbox box
					initially (the first time the form is shown to
					the user), <tt>false</tt> otherwise.
*/
public static final String checkbox(
 final HttpServletRequest req, final String name)
	{
	final String val = getParameter(req, name);
	if (dbg) System.out.println(name+"="+val);
	
	if (val == null) {
		return "";
		}
	
	//value != null (any value)
	return "checked";
	}

/**
Returns the submitted state of the specified check box which is part
of a checkbox group (multiple check buttons with the same name). For
non-group check boxes, use the {@link #checkbox checkbox} method instead.
<p>
Note: Checkboxes with no specific value are submitted with the value
"on" (case-insensitive) by the browser (as seen in the second and
third example below). Example:
<blockquote><pre>

<b><u>Good</u></b>:
&lt;input type=checkbox name=<font color=green>foo</font> <u>value='<font color=blue>somevalue1</font>'</u> [=State.checkboxgroup(req, "<font color=green>foo</font>", <font color=blue>"somevalue1"</font>)]&gt;
&lt;input type=checkbox name=<font color=green>foo</font> <u>value='<font color=blue>somevalue2</font>'</u> [=State.checkboxgroup(req, "<font color=green>foo</font>", <font color=blue>"somevalue2"</font>)]&gt;

<b>Don't do this</b>:
&lt;input type=checkbox name=<font color=green>foo</font> [=State.checkboxgroup(req, "<font color=green>foo</font>", <font color=blue>"on"</font>)]&gt;
&lt;input type=checkbox name=<font color=green>foo</font> [=State.checkboxgroup(req, "<font color=green>foo</font>", <font color=blue>"on"</font>)]&gt;
</pre></blockquote>
This method takes a <code>value</code> parameter that specifies the
value of the check box in the checkbox group. <u>This method
requires that each checkbox in the group have a specified and unique
value attribute</u>. 
<p>
<b>Note</b>: For check boxes, the state maintainence value is the
presence/absence of the <tt>checked</tt> attribute. The value
returned by this method will be either an empty string or
<tt>checked</tt>. This should not be assigned to the 'value'
attribute (if present) of the checkbox. Rather, it should be a
stand-alone attribute.
*/
public static final String checkboxgroup(
	final HttpServletRequest req, final String name, final String value)
	{
	final String[] val = getParameterValues(req, name);
	if (dbg) System.out.println(name+"="+Arrays.toString(val));
	
	if (val == null)
		return "";
	
	/*
	we cannot say:
	
		if (val[n].equals(value) || val[n].equalsIgnoreCase("on")) ...
	
	because a checkbox with no value - if checked - will also check all
	other checkboxes in the group (since we get foo=on and we don't
	know which checkbox to turn on in that case, since they are all 
	called foo)
	*/
	for (int n = 0; n < val.length; n++) 
		{
		if (val[n].equals(value)) {
			return "checked";
			}
		}
	
	return "";
	}


/**
Returns the value of the submitted hidden field or an empty string if no
value was submitted (this can happen if the user hacked/removed this
value manually).
Example:
<blockquote><pre>
&lt;input type=hidden name=<font color=green>foo</font> value='[=State.hidden(req, "<font color=green>foo</font>")]'&gt;
</pre></blockquote>
*/
public static final String hidden(final HttpServletRequest req, final String name)
	{
	return text(req, name);
	}


//<input type=image: get x,y co-ordinates of the click. not needed for form
//maintainence
//<input type=submit: not needed for form maintainence


/**
Returns the state of the submitted select. For select options, we
need to maintain whichever option(s) were selected by the user. The
presence/absence of the <tt>selected</tt> attribute maintains this
state.
<p>
When maintaining states for select input types:
<ol>
<li>
We know the select options on the server-side and write them
out dynamically (or statically) on the server-side and 
then send the page to the client. To maintain state, we 
say:
<blockquote>
<pre>
&lt;select name=<font color=green>sel1</font>&gt;
&lt;option value=<font color=red>"1"</font> <font color=blue>[=State.select(req, <font color=green>"sel1"</font>, <font color=red>"1")</font>]</font>&gt; Option 1
&lt;option           <font color=blue>[=State.select(req, <font color=green>"sel1"</font>, <font color=red>"Option 1"</font>)]</font>&gt; <font color=red>Option 1</font>
&lt;/select&gt;
</pre>
</blockquote>

<li>
We query and create select options on the client-side using
ajax/js/etc (for example, if the client chooses a country, we query
a database and show the states for that country via another
client-side select, created dynamically). Upon submission of this
form, we still need to maintain the state of this option. To do
this, we need to know, on the server side, the names of any new
selects that are created on the client (and/or the names of
initially empty selects that are re-populated on the client).
Suppose, initially we have:

<blockquote><pre>
&lt;select name=<font color=green>states</font>&gt;
&lt;/select&gt;
</pre></blockquote>

On the client, we then add:
<blockquote><pre>
&lt;select name=<font color=green>states</font>&gt;
&lt;option&gt; <font color=red>Pennsylvania</font> (generated on client)
&lt;/select&gt;
</pre></blockquote>

To maintain the state on the server for this option, we now have to 
say on the server-side (upon form submission):
<blockquote><pre>
&lt;select name=<font color=green>states</font>&gt;
   &lt;option <font color=blue>[=State.select(req, <font color=green>"states"</font>, </font><font color=red>"?"</font>)]&gt;<font color=red>Pennsylvania</font> (generated on client) 
&lt;select>
</pre></blockquote>

But we don't know what value to put in <tt><font color=red>"?"</font></tt>
on the server-side because <tt><font color=red>Pennsylvania</font></tt> was generated on the 
client. So we can do any of the following:
<style>ol ol li { padding-top: 1em; }</style>
<ol style="list-style: square">
<li>Rerun the query on the server, get the list of states again
and say, for example something like:
<blockquote>
<pre>
[[
if (...options were added by by client...)
    { //maintain state
    List option_list == ...create new {@link SelectOption}'s here [possibly via a query]...
    out.print(" &lt;select name=<font color=green>states</font>&gt; ");
    for (int n = 0; n < option_list.size(); n++) 
        {
        {@link SelectOption} opt = (SelectOption) option_list.get(n);
        String html = opt.getHTML();
        String val  = opt.getValue();
        out.print(" &lt;option val=");
        our.print(val);
        if (<font color=blue>State.select(req, <font color=green>"states"</font>, val)</font>) {
            out.print(" selected ");
            }
        out.print(" &gt; ");
        out.print(html);
        }
    out.print(" &lt;/select&gt; ");
    }
]]
</pre></blockquote>
But this (kind of) defeats the purpose of using ajax because we have to 
replicate the logic on the server. 
</li>
<li>
Another solution is to write out the possible values to be shown to
the client as a javascript array on the page. Then the client js can
show the contents of a an array depending on user choice. This saves
a ajax call to the database. If we store the results of the query in
a cache or a static variable, then we don't have to re-run the query
every time and can simply rewrite the array from the cached query
results.
</li>
<li>
Another option is that when the form is submitted to the server, it
sets a hidden field with the submitted value of the select. Upon
recieving the page, javascript on the browser re-runs the query on
the client side again and then uses the hidden field to restore the
state of the select option.
</li>
</ol>
<p>These issue only arise when select elements are being modified
on the client side via Javascript. 
</li>
</ol>

@param	req 			the HttpServletRequest 
@param	selectName		the name of the select field
@param	optionVal		the value attribute of the option. If the
						option has no option attribute, the html
						text associated with that option. Browsers
						do not submit leading spaces in the html
						text so, for example for: <tt>
						"&lt;option&gt;&nbsp;&nbsp;Option 1"</tt>, the
						value would be "Option 1"

*/
public static final String select(final HttpServletRequest req, 
				final String selectName, final String optionVal)
	{
	final String[] val = getParameterValues(req, selectName);
	if (dbg) System.out.println(selectName+"="+Arrays.toString(val));
	
	if (val == null)
		return "";
	
	for (int n = 0; n < val.length; n++) {
		if (dbg) System.out.println(val[n] + ";" + optionVal + ";" + (optionVal.equals(val[n]) ? "true/checked" : "false") );
		if (optionVal.equals(val[n])) {
			return "selected";
			}
		}
	
	return "";
	}


/**
Convenience method that creates a list of select options that
maintain form state. Example:
<blockquote><pre>
[[
List list = (..create a list of {@link SelectOption}, possible from database lookup..)
]]
&lt;select name=foo&gt;
[[ makeOptions(out, req, "foo", list); ]]
&lt;/select&gt;
</pre></blockquote>

@param	out			destination where the options will be printed
@param  req			the current HttpServletRequest
@param	selectName	the name of the select field
@param	optionsList	a list of objects of type {@link SelectOption}
*/
public static void makeOptions(final PrintWriter out,
 final HttpServletRequest req, 
 final String selectName, final List optionList)
	{
	final int size = optionList.size();
	for (int n = 0; n < size; n++) 
		{
		SelectOption opt = (SelectOption) optionList.get(n);
		String html = HTMLUtil.quoteToEntity(opt.getHTML());
		String val = opt.getValue();
		String escaped_val = HTMLUtil.quoteToEntity(val);
		out.print("<option val='");
		out.print(escaped_val);
		out.print("'");
		out.print(opt.selected() ? " selected " : " ");
		out.print(State.select(req, selectName, val));
		out.print(">");
		out.print(html);
		if ( (n + 1) < size) {
			out.print("\n");
			}
		}
	}

/**
Convenience method that creates a list of select options that
maintain form state. Returns the list of options as a string
(rather than printing it to a printwriter as some other variants of
this method do). Example:
<blockquote><pre>
[[
List list = (..create a list of {@link SelectOption}, possible from database lookup..)
]]
&lt;select name=foo&gt;
[= makeOptions(req, "foo", list); ]
&lt;/select&gt;
</pre></blockquote>

@param  req			the current HttpServletRequest
@param	selectName	the name of the select field
@param	optionsList	a list of objects of type {@link SelectOption}
*/
public static String makeOptions(final HttpServletRequest req, 
 final String selectName, final List optionList)
	{
	final int size = optionList.size();
	final StringBuilder buf = new StringBuilder(size * 32);
	for (int n = 0; n < size; n++) 
		{
		SelectOption opt = (SelectOption) optionList.get(n);
		String html = HTMLUtil.quoteToEntity(opt.getHTML());
		String val = opt.getValue();
		String escaped_val = HTMLUtil.quoteToEntity(val);
		buf.append("<option val='");
		buf.append(escaped_val);
		buf.append("'");
		buf.append(opt.selected() ? " selected " : " ");
		buf.append(State.select(req, selectName, val));
		buf.append(">");
		buf.append(html);
		if ( (n + 1) < size) {
			buf.append("\n");
			}
		}
	return buf.toString();
	}




/**
Sets the value of the form element with the specified name. This
method is useful to set initial values for the form. The initial
values shown to the user can be different per user, for example,
when the user is editing some data that already exists on the database.
<p>
This method can be invoked more than once. This is useful to set
multiple values for the same field (for example, a select box
with multiple selections or multiple checkboxes with the same name).
<blockquote><pre>
State.set(req, "checkbox1", "1.a");
State.set(req, "checkbox1", "1.b");
</pre></blockquote>
*/
public static final void set(
 final HttpServletRequest req, final String name, final String value)
	{
	final Object obj = req.getAttribute(name);
	
	if (obj == null)  {  //obj not already present
		req.setAttribute(name, value);
		return;
		}

	//obj is present. could be a single string or string[] (list)
	if (obj instanceof String) 
		{
		//defaults to size of 10 which is fine for us
		final ArrayList list = new ArrayList();
		
		list.add(obj);   //save old single string value
		list.add(value); //save new value
		req.setAttribute(name, list);
		}
	else if (obj instanceof ArrayList) {
		//append new string value to the list
		((ArrayList)obj).add(value);
		}
	else{ 
		throw new IllegalArgumentException("Only strings can be added. You are trying to add: " + obj + "/" + obj.getClass().getName());
		}
	}

private static final String cleared_key = "_fc.web.simpleforms.cleared";
	
/**
The servlet api is written by retarded monkeys smoking crack....which 
is why there is no way to modify/remove/change/clear 
the parameters in the servlet request (the parameters map is read-only)
..which leads to all sort of problems in html form processing.
<p>
This method changes the state of <b>all</b> fields such that methods
that return state (text, textarea, radio etc) will act as if the
form field had not been submitted by the user. However, any values
set by the {@link set} method <i>after</i> invoking this method
<i>will</i> be seen.
<p>
All this is really useful when form data has been saved and the
user needs to be shown the same form, but with an empty (fresh)
form state. So for example:
<blockquote><pre>
// At the top of a page
String field_1 = req.getParameter("field_1");
String field_2 = req.getParameter("field_2");

Errors err =  validate_and_save(field1, field2);
if (err == null) {
    State.<font color=blue>clear</font>(req);  //req is the HttpServletRequest object
    State.set("field_1", "enter a value"); //initial value for field_1
    }

&lt;form&gt;
[[ if (err != null) { err.renderFormErrors(out); } ]]
&lt;input type=text name=field_1 value='[=State.text(req, "field_1")]'&gt;
&lt;input type=text name=field_2 value='[=State.text(req, "field_2")]'&gt;
&lt;/form&gt;
</pre></blockquote>
Note: redirecting the client to the form page also has the same
effect (of clearing request parameters), because that forces a 
entirely new request object to be used on the server.
*/
public static final void clear(final HttpServletRequest req)
	{
	Enumeration e = req.getAttributeNames();
	while (e.hasMoreElements()) {
		req.removeAttribute((String)e.nextElement());
		}
		
	req.setAttribute(cleared_key, "");
	}
	
// ----------- escaped ----------------
/**
Convenience method that returns: <tt>{@link 
fc.util.HTMLUtil.quoteToEntity HTMLUtil.quoteToEntity}(text(req, name));</tt>
Useful for setting arbitrary values in the value="..." part 
of a text field.
*/
public static final String escapedText(
final HttpServletRequest req, final String name)
	{
	return HTMLUtil.quoteToEntity(text(req, name));
	}

/**
Convenience method that returns: <tt>{@link 
fc.util.HTMLUtil.quoteToEntity(String) HTMLUtil.quoteToEntity}(password (req, name));</tt>
Useful for setting arbitrary values in the value="..." part 
of a password field.
*/
public static final String escapedPassword(
final HttpServletRequest req, final String name)
	{
	return HTMLUtil.quoteToEntity(password(req, name));
	}


/**
Convenience method that returns: <tt>{@link 
fc.util.HTMLUtil.quoteToEntity(String) HTMLUtil.quoteToEntity}(hidden(req, name));</tt>
Useful for setting arbitrary values in the value="..." part 
of a hidden field.
*/
public static final String escapedHidden(
final HttpServletRequest req, final String name)
	{
	return HTMLUtil.quoteToEntity(hidden(req, name));
	}


private static final String disabled_key = "_fc.web.simpleforms.disabled";

/**
Convenience method that marks the specified field as disabled. This
value can be later retrieved while rendering the form via the
getDisabled or disabled methods.
*/
public static final void setDisabled(
 final HttpServletRequest req, final String name)
	{
	Set set = (Set) req.getAttribute(disabled_key);
	if (set == null) {
		set = new HashSet();
		req.setAttribute(disabled_key, set);
		}
	set.add(name);
	}

/**
Returns true if the specified field was marked as disabled. 
Example usage:
<blockquote><pre>
[= State.getDisabled(req, "somefield") ? "disabled" : ""]
</pre></blockquote>
*/
public static final boolean getDisabled(
 final HttpServletRequest req, final String name)
	{
	Set set = (Set) req.getAttribute(disabled_key);
	if (set == null) {
		return false;
		}
	return set.contains(name);
	}

/**
Convenience method. Instead of saying:
Example usage:
<blockquote><pre>
[= State.getDisabled(req, "somefield") ? "disabled" : ""]
</pre></blockquote>
this method allows one to say:
<blockquote><pre>
[= State.disabled(req, "somefield")]
</pre></blockquote>
*/
public static final String disabled(
 final HttpServletRequest req, final String name)
	{
	if (getDisabled(req, name))
		return "disabled";
	else 
		return "";
	}

} //~class State