// Copyright (c) 2001 Hursh Jain (http://www.mollypages.org) 
// The Molly framework is freely distributable under the terms of an
// MIT-style license. For details, see the molly pages web site at:
// http://www.mollypages.org/. Use, modify, have fun !

package fc.web.simpleforms;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.*;
import java.util.regex.*;

import fc.jdbc.*;
import fc.io.*;
import fc.util.*;

/**
Utility methods for common validation needs.

@author hursh jain
**/
public final class Valid
{
final 	static boolean 		 dbg = false; //internal debugging
	 	static boolean 		 trim = true;
private static final String  intOnlyRE_str 		= "\\s*\\d*\\s*";
private static final Pattern intOnlyRE 			= Pattern.compile(intOnlyRE_str); 
private static final String  isFloatOnlyRE_str	= "\\s*-?\\d*\\.?\\d*\\s*";
private static final Pattern isFloatOnlyRE		= Pattern.compile(isFloatOnlyRE_str);
	  
/**
If set to <tt>true</tt>, methods in this class will trim the string 
entered by the user before attempting to run further validation on it,
else strings will not be trimmed automatically. Defaults to 
<tt>true</tt>.
*/
public static void setTrimSpaces(boolean val)
	{
	trim = val;
	}

/**
Ensures that the string is composed of only an digits [0-9] with
optional leading/trailing blanks. (floating point/real numbers are
not allowed because they contain a ".")
*/
public static boolean isIntegerOnly(String str) 
	{
	if (str == null)
		return false;
		
	if (trim) {
		str = str.trim();
		}
		
	return intOnlyRE.matcher(str).matches();
	}

/**
Ensures that the string is composed of only floating point
numbers (digits [0-9] and maybe a  "."), with optional 
leading/trailing blanks. Negative numbers (with a leading "-")
are also allowed.
*/
public static boolean isFloatOnly(String str) 
	{
	if (str == null)
		return false;
		
	if (trim) {
		str = str.trim();
		}
		
	return isFloatOnlyRE.matcher(str).matches();
	}


public static boolean isEmpty(String str) 
	{
	if (str == null) {
		return false;
		}
	
	if (trim) {
		str = str.trim();
		}
	
	if (str.length() == 0) {
		return true;
		}
		
	return false;
	}

/**
Returns <tt>true</tt> is the specified string is not <tt>null</tt> <b>and</b> 
not empty (<tt>""</tt>).
*/
public static boolean isNotEmpty(String str) 
	{
	if (str == null) {
		return false;
		}
	
	if (trim) {
		str = str.trim();
		}
	
	if (str.length() == 0) {
		return false;
		}

	return true;
	}

/**
Returns true if the string is non-empty and it's length
is between the specified min and max lengths (both inclusive).
*/
public static boolean length(String str, final int min, final int max) 
	{
	if (str == null) {
		return false;
		}

	if (trim) {
		str = str.trim();
		}
		
	final int len = str.length();
	if (len < min || len > max) {
		return false;
		}

	return true;
	}

/**
Returns <tt>true</tt> if the specified string only contains the
allowed characters. If the string contains any other characters
that are not allowed, returns <tt>false</tt>
*/
public static boolean isAllowed(String str, final String allowedChars)
	{
	if (str == null)
		return false;

	if (trim) {
		str = str.trim();
		}
	
	for (int n = 0; n < str.length(); n++) 
		{
		char c = str.charAt(n);
		boolean found = false;
		for (int k = 0; k < allowedChars.length(); k++) 
			{
			char mchar = allowedChars.charAt(k);
			if (c == mchar) {
				found = true;
				}
			}
		if (! found) {
			return false;
			}
		}
	
	return true;
	}

/**
Returns <tt>true</tt> if the specified string has no unallowed
characters. If the string contains any unallowed characters, returns
<tt>false</tt>
*/
public static boolean isNotAllowed(String str, final String unallowedChars)
	{
	if (str == null)
		return false;

	if (trim) {
		str = str.trim();
		}
	
	for (int n = 0; n < str.length(); n++) 
		{
		char c = str.charAt(n);
		boolean passed = true;
		for (int k = 0; k < unallowedChars.length(); k++) 
			{
			char mchar = unallowedChars.charAt(k);
			if (c == mchar) {
				passed = false;
				}
			}
		if (! passed) {
			return false;
			}
		}
	
	return true;
	}

public static void main (String args[])
	{
	p("Valid.isIntegerOnly(\" 3243 \")", Valid.isIntegerOnly(" 3243 "), true );  //trim
	p("Valid.isIntegerOnly(\" 3243. \")",  Valid.isIntegerOnly(" 3243. "), false );
	
	p( "Valid.isFloatOnly(\" 3243. \")", Valid.isFloatOnly(" 3243. "), true );
	p( "Valid.isFloatOnly(\" -3243. \")", Valid.isFloatOnly(" -3243. "), true );
	p( "Valid.isFloatOnly(\" -3243.003 \")", Valid.isFloatOnly(" -3243.003 "), true );
	p( "Valid.isFloatOnly(\".3243.0\")", Valid.isFloatOnly(".3243.0"), false );
	p( "Valid.isFloatOnly(\"-32dd43.003\")", Valid.isFloatOnly("-32dd43.003"), false );
	
	p( "Valid.isEmpty(\"  \")", Valid.isEmpty("  "), true); //if trim
	p( "Valid.isEmpty(\"\")", Valid.isEmpty(""), true);
	
	p( "Valid.length(\" abc  \", 1, 3)", Valid.length(" abc  ", 1, 3), true ); //trim
	Valid.setTrimSpaces(false);
	p( "Valid.length(\" abc  \", 1, 3)", Valid.length(" abc  ", 1, 3), false ); 
	Valid.setTrimSpaces(true);
		
	p( "Valid.isAllowed\"abc\", \"abc\")", Valid.isAllowed("abc", "abc"), true );
	p( "Valid.isAllowed(\"abc\", \"abcdef\")", Valid.isAllowed("abc", "abcdef"), true );
	p( "Valid.isAllowed(\"abc\", \"ab\")", Valid.isAllowed("abc", "ab"), false );
	p( "Valid.isAllowed(\"abc\", \"\")", Valid.isAllowed("abc", ""), false );
	
	p( "Valid.isNotAllowed(\"abc\", \"4\")", Valid.isNotAllowed("abc", "4"), true ); 
	p( "Valid.isNotAllowed(\"abc\", \"#!@9\")", Valid.isNotAllowed("abc", "#!@9"), true ); 
	p( "Valid.isNotAllowed(\"abc4\", \"\")", Valid.isNotAllowed("abc4", ""), true ); 
	p( "Valid.isNotAllowed(\"abc4\", \"a\")", Valid.isNotAllowed("abc4", "a"), false ); 
	
	if (_badcount == 0)
		System.out.println("All internal tests were successful..");
	else
		System.out.println(_badcount + " tests failed..");
	}
	
private static int _badcount = 0;
private static void p(String s, boolean a, boolean b)
	{
	if (a != b) {
		_badcount++;
		System.out.println("FAILED: " + s);
		}
	}
}          //~class Valid
